package ru.yandex.solomon.project.manager.api.v3.intranet.rest;

import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.swagger.annotations.Api;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.api.v3.project.manager.CreateProjectBySubjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.CreateProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.DeleteProjectBySubjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.DeleteProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.GetProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectOperationsResponse;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectsRequest;
import ru.yandex.monitoring.api.v3.project.manager.Role;
import ru.yandex.monitoring.api.v3.project.manager.UpdateProjectBySubjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.UpdateProjectRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.operation.OperationDtoConverter;
import ru.yandex.solomon.project.manager.api.WhiteListAuthorizer;
import ru.yandex.solomon.project.manager.api.util.ProtoJsonUtils;
import ru.yandex.solomon.project.manager.api.v3.intranet.ProjectService;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
@Api(tags = "projects")
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class ProjectsController {

    private final ProjectService service;
    private final OperationDtoConverter operationConverter;
    private final WhiteListAuthorizer whiteListAuthorizer;

    public ProjectsController(
            ProjectService projectService,
            @Qualifier("whiteListAuthorizer") WhiteListAuthorizer whiteListAuthorizer)
    {
        this.service = projectService;
        this.whiteListAuthorizer = whiteListAuthorizer;
        operationConverter = new OperationDtoConverter();
    }

    @GetMapping(path = "/api/v3/projects/{projectId}")
    CompletableFuture<String> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty) {
        GetProjectRequest getRequest = GetProjectRequest.newBuilder()
                .setProjectId(projectId)
                .build();
        return service.get(getRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @GetMapping(path = "/api/v3/projects")
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "filterByName", defaultValue = "") String filterByName,
            @RequestParam(value = "filterByAbcService", defaultValue = "") String filterByAbcService,
            @RequestParam(value = "filterByRole", required = false) Set<Role> filterByRole,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty) {
        ListProjectsRequest listRequest = ListProjectsRequest.newBuilder()
                .setFilterByName(filterByName)
                .setFilterByAbcService(filterByAbcService)
                .addAllFilterByRole(Nullables.orEmpty(filterByRole))
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();
        return service.list(listRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @PostMapping(path = "/api/v3/projects/{projectId}", consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        return CompletableFutures.safeCall(() -> {
            CreateProjectRequest.Builder builder = CreateProjectRequest.newBuilder();
            ProtoJsonUtils.fromJson(body, builder);
            builder.setProjectId(projectId);
            CreateProjectRequest request = builder.build();

            return service.create(request, subject);
        })
                .handle(operationConverter::wrap)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @PostMapping(path = "/api/v3/projectsBySubject/{projectId}", consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> createBySubject(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        return CompletableFutures.safeCall(() -> {
            CreateProjectBySubjectRequest.Builder builder = CreateProjectBySubjectRequest.newBuilder();
            ProtoJsonUtils.fromJson(body, builder);
            builder.setCreateProjectRequest(builder.getCreateProjectRequest().toBuilder()
                    .setProjectId(projectId)
                    .build());
            CreateProjectBySubjectRequest request = builder.build();
            AuthSubject targetSubject = whiteListAuthorizer.authorize(subject, request.getSubject());

            return service.create(request.getCreateProjectRequest(), targetSubject);
        })
                .handle(operationConverter::wrap)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @PatchMapping(path = "/api/v3/projects/{projectId}", consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        return CompletableFutures.safeCall(() -> {
            UpdateProjectRequest.Builder builder = UpdateProjectRequest.newBuilder();
            ProtoJsonUtils.fromJson(body, builder);
            builder.setProjectId(projectId);
            UpdateProjectRequest request = builder.build();

            return service.update(request, subject);
        })
                .handle(operationConverter::wrap)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @PatchMapping(path = "/api/v3/projectsBySubject/{projectId}", consumes = MediaType.APPLICATION_JSON_VALUE)
    CompletableFuture<String> updateBySubject(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        return CompletableFutures.safeCall(() -> {
            UpdateProjectBySubjectRequest.Builder builder = UpdateProjectBySubjectRequest.newBuilder();
            ProtoJsonUtils.fromJson(body, builder);
            builder.setUpdateProjectRequest(builder.getUpdateProjectRequest().toBuilder()
                    .setProjectId(projectId)
                    .build());
            UpdateProjectBySubjectRequest request = builder.build();
            AuthSubject targetSubject = whiteListAuthorizer.authorize(subject, request.getSubject());

            return service.update(request.getUpdateProjectRequest(), targetSubject);
        })
                .handle(operationConverter::wrap)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @DeleteMapping(path = "/api/v3/projects/{projectId}")
    CompletableFuture<String> delete(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @PathVariable("projectId") String projectId)
    {
        return CompletableFutures.safeCall(() -> {
            DeleteProjectRequest.Builder builder = DeleteProjectRequest.newBuilder();
            builder.setProjectId(projectId);
            DeleteProjectRequest request = builder.build();

            return service.delete(request, subject);
        })
                .handle(operationConverter::wrap)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @DeleteMapping(path = "/api/v3/projectsBySubject/{projectId}")
    CompletableFuture<String> deleteBySubject(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @PathVariable("projectId") String projectId,
            @RequestBody byte[] body)
    {
        return CompletableFutures.safeCall(() -> {
            DeleteProjectBySubjectRequest.Builder builder = DeleteProjectBySubjectRequest.newBuilder();
            ProtoJsonUtils.fromJson(body, builder);
            builder.setDeleteProjectRequest(builder.getDeleteProjectRequest().toBuilder()
                    .setProjectId(projectId)
                    .build());
            DeleteProjectBySubjectRequest request = builder.build();
            AuthSubject targetSubject = whiteListAuthorizer.authorize(subject, request.getSubject());

            return service.delete(request.getDeleteProjectRequest(), targetSubject);
        })
                .handle(operationConverter::wrap)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @GetMapping(path = "/api/v3/projects/{projectId}/operations")
    CompletableFuture<String> getOperations(
            @RequireAuth AuthSubject subject,
            @PathVariable("projectId") String projectId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        return CompletableFuture.completedFuture(ProtoJsonUtils.toJson(ListProjectOperationsResponse.getDefaultInstance(), pretty));
    }
}
