package ru.yandex.solomon.project.manager.spring;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillClose;

import io.grpc.Server;
import io.grpc.ServerInterceptor;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.grpc.utils.GrpcServerFactory;
import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.grpc.utils.RequestParamsInterceptor;
import ru.yandex.solomon.auth.Authenticator;
import ru.yandex.solomon.auth.grpc.AuthenticationInterceptor;
import ru.yandex.solomon.auth.grpc.GrpcRequestLoggingInterceptor;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcServerConfig;
import ru.yandex.solomon.core.grpc.GrpcContext;
import ru.yandex.solomon.project.manager.api.ApiContext;

/**
 * @author Alexey Trushkin
 */
@Configuration
@Import({
        GrpcContext.class,
        ApiContext.class,
})
@SuppressWarnings({"SpringJavaInjectionPointsAutowiringInspection", "OptionalUsedAsFieldOrParameterType"})
@ParametersAreNonnullByDefault
public class GrpcApiContext implements DisposableBean {

    private final GrpcServerFactory grpcServerFactory;
    @WillClose
    @Nullable
    private Server privateApiServer;

    @Autowired
    public GrpcApiContext(GrpcServerFactory grpcServerFactory) {
        this.grpcServerFactory = grpcServerFactory;
    }

    @Bean
    public AuthenticationInterceptor authInterceptor(Authenticator authenticator) {
        return new AuthenticationInterceptor(authenticator);
    }

    @Bean
    public Server privateApiMonitoringGrpcServer(
            @Qualifier("GrpcServerConfig") TGrpcServerConfig grpcServerConfig,
            AuthenticationInterceptor authenticationInterceptor,
            @Autowired(required = false) List<GrpcService> grpcServices
    ) {
        List<ServerInterceptor> interceptors = List.of(
                new GrpcRequestLoggingInterceptor(),
                authenticationInterceptor,
                new RequestParamsInterceptor());
        this.privateApiServer = this.grpcServerFactory.makePublicServer(
                "GrpcServerConfig",
                grpcServerConfig,
                interceptors,
                grpcServices);
        grpcServerFactory.startWithTimeout(grpcServerConfig, privateApiServer);
        return privateApiServer;
    }

    @Override
    public void destroy() {
        Server privateServer = this.privateApiServer;
        if (privateServer != null) {
            this.privateApiServer = null;
            privateServer.shutdown();
        }
    }
}
