package ru.yandex.solomon.project.manager.spring;

import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.function.Supplier;

import com.google.common.net.HostAndPort;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.cloud.token.IamTokenClientOptions;
import ru.yandex.cloud.token.Jwt;
import ru.yandex.cloud.token.grpc.GrpcIamTokenClient;
import ru.yandex.discovery.DiscoveryServices;
import ru.yandex.juggler.client.HttpJugglerProjectChangesClient;
import ru.yandex.juggler.client.JugglerProjectChangesClient;
import ru.yandex.juggler.config.JugglerClientOptions;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.passport.tvmauth.TvmClient;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.config.IamKeyJson;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.config.protobuf.project.manager.JugglerConfig;
import ru.yandex.solomon.config.protobuf.project.manager.SolomonGatewayConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.project.manager.api.v3.intranet.impl.listener.JugglerProjectChangeListener;
import ru.yandex.solomon.project.manager.api.v3.intranet.impl.listener.SolomonProjectChangeListener;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.util.SolomonEnv;

/**
 * @author Alexey Trushkin
 */
@Configuration
public class ProjectChangeListenerContext {

    @Bean
    public SolomonProjectChangeListener solomonProjectChangeListener(
            ThreadPoolProvider threadPoolProvider,
            MetricRegistry registry,
            SolomonGatewayConfig solomonGatewayConfig,
            @Qualifier("gatewayTvmClient") Optional<TvmClient> tvmClient,
            @Qualifier("iamTokenProvider") Optional<TokenProvider> iamTokenProvider)
    {
        if (tvmClient.isEmpty() && iamTokenProvider.isEmpty()) {
            return null;
        }
        Supplier<String> headerSupplier;
        Supplier<String> headerValueSupplier;
        if (tvmClient.isPresent()) {
            headerValueSupplier = () -> tvmClient.get().getServiceTicketFor(solomonGatewayConfig.getDestinationTvmClient());
            headerSupplier = AuthType.TvmService::getHeaderName;
        } else {
            headerValueSupplier = () -> iamTokenProvider.get().getToken();
            headerSupplier = AuthType.IAM::getHeaderName;
        }
        return new SolomonProjectChangeListener(headerSupplier,
                headerValueSupplier,
                registry,
                solomonGatewayConfig,
                threadPoolProvider.getExecutorService("CpuLowPriority", ""));
    }

    @Bean
    public JugglerProjectChangesClient jugglerProjectChangesClient(
            Optional<JugglerConfig> jugglerConfig,
            ThreadPoolProvider threadPoolProvider,
            @Qualifier("jugglerTvmClient") Optional<TvmClient> tvmClient,
            MetricRegistry metricRegistry)
    {
        if (tvmClient.isEmpty() || jugglerConfig.isEmpty()) {
            return null;
        }
        JugglerClientOptions opts = JugglerClientOptions.newBuilder()
                .setUrl(jugglerConfig.get().getApiUrl())
                .setMetricRegistry(metricRegistry)
                .setExecutor(threadPoolProvider.getExecutorService("CpuLowPriority", ""))
                .setTokenProvider(() ->
                        AuthType.TvmService.getValuePrefix() + tvmClient.get().getServiceTicketFor(jugglerConfig.get().getDestinationTvmClient()))
                .setTokenHeaderProvider(AuthType.TvmService::getHeaderName)
                .build();
        return new HttpJugglerProjectChangesClient(opts);
    }

    @Bean
    public JugglerProjectChangeListener jugglerProjectChangeListener(
            Optional<JugglerProjectChangesClient> client,
            FeatureFlagsHolder featureFlagsHolder)
    {
        if (client.isEmpty()) {
            return null;
        }
        return new JugglerProjectChangeListener(client.get(), featureFlagsHolder);
    }

    @Bean(name = "iamTokenProvider")
    public TokenProvider tokenProvider(
            SolomonGatewayConfig solomonGatewayConfig,
            Optional<IamTokenClient> iamTokenClient,
            TAuthConfig authConfig,
            ThreadPoolProvider threads,
            SecretProvider secretProvider)
    {
        if (!solomonGatewayConfig.getIamAuth()) {
            return null;
        }
        if (!authConfig.hasIamConfig() || iamTokenClient.isEmpty()) {
            return null;
        }
        var jwtBuilder = Jwt.newBuilder()
            .withTtl(Duration.ofHours(1));

        switch (authConfig.getIamConfig().getKeyTypeCase()) {
            case KEY_AUTH -> {
                var keyAuth = authConfig.getIamConfig().getKeyAuth();
                var keyFilePath = Path.of(keyAuth.getKeyPath());
                if (SolomonEnv.DEVELOPMENT.isActive() && !Files.exists(keyFilePath)) {
                    return TokenProvider.of("fake-token");
                }
                jwtBuilder.withAccountId(keyAuth.getAccountId())
                        .withKeyId(keyAuth.getKeyId())
                        .withPrivateKey(keyFilePath);
            }
            case KEY_JSON_AUTH -> {
                Optional<String> keyData = secretProvider.getSecret(authConfig.getIamConfig().getKeyJsonAuth().getKeyData());
                if (SolomonEnv.DEVELOPMENT.isActive() && keyData.isEmpty()) {
                    return TokenProvider.of("fake-token");
                }

                IamKeyJson iamKey = IamKeyJson.parseFromJson(
                        keyData.orElseThrow(() -> new RuntimeException("cannot create auth provider with empty private key")));

                jwtBuilder.withAccountId(iamKey.getAccountId())
                        .withKeyId(iamKey.getId())
                        .withPrivateKey(iamKey.getPrivateKey());
            }
        }
        return TokenProvider.iam(iamTokenClient.get(), jwtBuilder, threads.getSchedulerExecutorService());
    }

    @Bean
    public IamTokenClient iamTokenClient(
            SolomonGatewayConfig solomonGatewayConfig,
            Optional<TAuthConfig> config,
            ThreadPoolProvider threads)
    {
        if (!solomonGatewayConfig.getIamAuth() || config.isEmpty() || !config.get().hasIamConfig()) {
            return null;
        }
        ExecutorService executor = threads.getExecutorService(
                config.get().getIamConfig().getThreadPoolName(),
                "CloudConfig.IamTokenClient.ThreadPoolName");

        HostAndPort address = DiscoveryServices.resolve(config.get().getIamConfig().getTokenServiceAddresses()).get(0);
        return new GrpcIamTokenClient(IamTokenClientOptions.forAddress(address.getHost(), address.getPort())
                .withHandlerExecutor(executor)
                .withUserAgent("SolomonProjectManager"));
    }

}
