package ru.yandex.solomon.project.manager.spring;

import java.util.Optional;
import java.util.concurrent.Executor;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.TAbcClientConfig;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.db.dao.ClusterFlagsDao;
import ru.yandex.solomon.core.db.dao.ProjectFlagsDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServiceFlagsDao;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.dao.ShardFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbClusterFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServiceFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServiceProvidersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardFlagsDao;
import ru.yandex.solomon.project.manager.abc.db.AbcToProjectMappingDao;
import ru.yandex.solomon.project.manager.abc.db.ydb.YdbAbcToProjectMappingDao;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.ydb.YdbClients;

/**
 * @author Alexey Trushkin
 */
@Configuration
public class YdbContext implements DisposableBean {

    private final TKikimrClientConfig config;
    private final MetricRegistry registry;
    private final YdbClients ydbClients;
    private final Executor executor;

    public YdbContext(
            @Qualifier("CrossDcKikimrClientConfig") TKikimrClientConfig config,
            ThreadPoolProvider threadPoolProvider,
            MetricRegistry registry,
            Optional<IamTokenClient> iamTokenClient,
            SecretProvider secrets)
    {
        this.config = config;
        this.registry = registry;
        this.ydbClients = new YdbClients("CrossDcKikimrClientConfig", config, threadPoolProvider, registry, iamTokenClient, secrets);
        this.executor = threadPoolProvider.getExecutorService("CpuLowPriority", "");
    }

    @Bean("globalSchemeClient")
    SchemeClient schemeClient() {
        return ydbClients.getSchemeClient();
    }

    @Bean("globalTableClient")
    TableClient tableClient() {
        return ydbClients.getTableClient();
    }

    @Bean("globalTransport")
    RpcTransport ydbTransport() {
        return ydbClients.getTransport();
    }

    @Bean(name = "globalClientMetrics")
    MetricSupplier tableClientMetrics() {
        return ydbClients.getTableClientMetrics();
    }

    @Bean(name = "configKikimrRootPath")
    public String ydbRootPath() {
        return config.getSchemaRoot();
    }

    @Bean
    public ProjectsDao projectsDao(ObjectMapper objectMapper) {
        String tablePath = ydbClients.getSchemaRoot() + "/Config/V2/Project";
        var dao = new YdbProjectsDao(ydbClients.getTableClient(), tablePath, objectMapper, executor);
        return measure(dao, ProjectsDao.class);
    }

    @Bean
    public ServiceProvidersDao ydbServiceProviders(ObjectMapper objectMapper) {
        var path = ydbClients.getSchemaRoot() + "/Config/V2/ServiceProvider";
        var dao = new YdbServiceProvidersDao(ydbClients.getTableClient(), path, objectMapper);
        dao.migrateSchema();
        return measure(dao, ServiceProvidersDao.class);
    }

    @Bean
    public ProjectFlagsDao projectFlagsDao() {
        String tablePath = ydbClients.getSchemaRoot() + "/Config/V2/ProjectFlags";
        var dao = new YdbProjectFlagsDao(ydbClients.getTableClient(), tablePath);
        return measure(dao, ProjectFlagsDao.class);
    }

    @Bean
    public ClusterFlagsDao clusterFlagsDao() {
        String tablePath = ydbClients.getSchemaRoot() + "/Config/V2/ClusterFlags";
        var dao = new YdbClusterFlagsDao(ydbClients.getTableClient(), tablePath);
        return measure(dao, ClusterFlagsDao.class);
    }

    @Bean
    public ShardFlagsDao shardFlagsDao() {
        String tablePath = ydbClients.getSchemaRoot() + "/Config/V2/ShardFlags";
        var dao = new YdbShardFlagsDao(ydbClients.getTableClient(), tablePath);
        return measure(dao, ShardFlagsDao.class);
    }

    @Bean
    public ServiceFlagsDao serviceFlagsDao() {
        String tablePath = ydbClients.getSchemaRoot() + "/Config/V2/ServiceFlags";
        var dao = new YdbServiceFlagsDao(ydbClients.getTableClient(), tablePath);
        return measure(dao, ServiceFlagsDao.class);
    }

    @Bean
    public AbcToProjectMappingDao abcToProjectMappingDao(Optional<TAbcClientConfig> configOptional) {
        if (configOptional.isEmpty()) {
            return null;
        }
        String tablePath = ydbClients.getSchemaRoot() + "/Config/V2";
        var dao = new YdbAbcToProjectMappingDao(tablePath, ydbClients.getTableClient(), ydbClients.getSchemeClient());
        dao.createSchemaForTests();
        return measure(dao, AbcToProjectMappingDao.class);
    }

    private <TDao> TDao measure(TDao dao, Class<TDao> daoClass) {
        return DaoMetricsProxy.of(dao, daoClass, registry);
    }

    @Override
    public void destroy() {
        ydbClients.close();
    }

}
