package ru.yandex.solomon.project.manager.spring.cloud;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.ExecutorService;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.resourcemanager.GrpcResourceManagerClient;
import ru.yandex.cloud.resourcemanager.ResourceManagerClient;
import ru.yandex.cloud.resourcemanager.ResourceManagerClientOptions;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.cloud.resource.resolver.CloudProjectNamesUpdater;
import ru.yandex.solomon.config.protobuf.ResourceManagerClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.flags.FeatureFlagsHolder;

/**
 * @author Alexey Trushkin
 */
@Configuration
public class ProjectUpdaterContext {

    private static final Duration DEFAULT_CONNECT_TIMEOUT = Duration.ofSeconds(5);
    private static final Duration DEFAULT_REQUEST_TIMEOUT = Duration.ofSeconds(30);

    @Bean
    public CloudProjectNamesUpdater cloudProjectNamesUpdater(
            ThreadPoolProvider tpProvider,
            Optional<ResourceManagerClient> resourceManagerClient,
            ProjectsDao projectsDao,
            FeatureFlagsHolder featureFlagsHolder,
            @Qualifier("ClientId") Optional<String> clientId
    ) {
        if (resourceManagerClient.isEmpty()) {
            return null;
        }
        return new CloudProjectNamesUpdater(tpProvider, resourceManagerClient.get(), projectsDao, featureFlagsHolder, clientId);
    }

    @Bean
    public ResourceManagerClient resourceManagerClient(
            Optional<ResourceManagerClientConfig> configOptional,
            @Qualifier("iamTokenProvider") Optional<TokenProvider> iamTokenProvider,
            ThreadPoolProvider threads,
            MetricRegistry registry)
    {
        if (configOptional.isEmpty() || iamTokenProvider.isEmpty()) {
            return null;
        }
        ResourceManagerClientConfig config = configOptional.get();
        ExecutorService executor = threads.getExecutorService(
                config.getThreadPoolName(),
                "CloudConfig.ResourceManagerClient.ThreadPoolName");
        return new GrpcResourceManagerClient(ResourceManagerClientOptions.forAddress(config.getHost(), config.getPort())
                .withUserAgent("SolomonProjectManager")
                .withConnectTimeout(DEFAULT_CONNECT_TIMEOUT)
                .withRequestTimeout(DEFAULT_REQUEST_TIMEOUT)
                .withHandlerExecutor(executor)
                .withTokenProvider(iamTokenProvider.get())
                .withRegistry(registry)
                .withResolveExistingOnly(config.getResolveExistingOnly()));
    }

}

