package ru.yandex.solomon.project.manager.api;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Context;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monitoring.api.v3.project.manager.Subject;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.iam.IamSubject;
import ru.yandex.solomon.auth.local.AsUserSubject;
import ru.yandex.solomon.auth.oauth.OAuthSubject;
import ru.yandex.solomon.auth.openid.OpenIdSubject;
import ru.yandex.solomon.auth.sessionid.SessionIdAuthSubject;
import ru.yandex.solomon.auth.tvm.TvmSubject;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.auth.grpc.AuthenticationInterceptor.AUTH_CONTEXT_KEY;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class WhiteListAuthorizerTest {

    private WhiteListAuthorizer authorizer;

    @Before
    public void setUp() {
        authorizer = new WhiteListAuthorizer(Set.of("me"));
    }

    @Test(expected = AuthorizationException.class)
    public void cantAuth() {
        Context authContext =
                Context.current().withValue(AUTH_CONTEXT_KEY, new AsUserSubject("fail"));
        authContext.attach();
        authorizer.authorizeSupplier(Subject.newBuilder()
                .setAnonymousAccount(Subject.AnonymousSubject.newBuilder().build())
                .build()).get();
        Context.current().detach(authContext);
    }

    @Test
    public void auth() {
        Context authContext =
                Context.current().withValue(AUTH_CONTEXT_KEY, new AsUserSubject("me"));
        authContext.attach();
        assertEquals(AnonymousAuthSubject.class, authorizer.authorizeSupplier(Subject.newBuilder()
                .setAnonymousAccount(Subject.AnonymousSubject.newBuilder().build())
                .build()).get().getClass());
        assertEquals(IamSubject.class, authorizer.authorizeSupplier(Subject.newBuilder()
                .setIamUserSubject(Subject.IamUserSubject.newBuilder()
                        .setId("1")
                        .build())
                .build()).get().getClass());
        assertEquals(IamSubject.class, authorizer.authorizeSupplier(Subject.newBuilder()
                .setIamServiceSubject(Subject.IamServiceSubject.newBuilder()
                        .setId("1")
                        .build())
                .build()).get().getClass());
        assertEquals(TvmSubject.UserSubject.class, authorizer.authorizeSupplier(Subject.newBuilder()
                .setTvmUserSubject(Subject.TvmUserSubject.newBuilder()
                        .setLogin("1")
                        .setUid(11)
                        .build())
                .build()).get().getClass());
        assertEquals(TvmSubject.ServiceSubject.class, authorizer.authorizeSupplier(Subject.newBuilder()
                .setTvmServiceSubject(Subject.TvmServiceSubject.newBuilder()
                        .setClientId(11)
                        .build())
                .build()).get().getClass());
        assertEquals(OAuthSubject.class, authorizer.authorizeSupplier(Subject.newBuilder()
                .setOAuthSubject(Subject.OAuthSubject.newBuilder()
                        .setLogin("1")
                        .setUid(11)
                        .build())
                .build()).get().getClass());
        assertEquals(OpenIdSubject.class, authorizer.authorizeSupplier(Subject.newBuilder()
                .setOpenIdSubject(Subject.OpenIdSubject.newBuilder()
                        .setLogin("1")
                        .setAccountId("2")
                        .build())
                .build()).get().getClass());
        assertEquals(SessionIdAuthSubject.class, authorizer.authorizeSupplier(Subject.newBuilder()
                .setSessionIdCookieSubject(Subject.SessionIdCookieSubject.newBuilder()
                        .setLogin("1")
                        .build())
                .build()).get().getClass());
        Context.current().detach(authContext);
    }

}
