package ru.yandex.solomon.project.manager.api.v3.intranet;

import java.util.concurrent.CompletionException;
import java.util.concurrent.ForkJoinPool;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Throwables;
import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

import ru.yandex.monitoring.api.v3.project.manager.GetAbcDefaultProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.Project;
import ru.yandex.monitoring.api.v3.project.manager.SetAbcDefaultProjectRequest;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.kikimr.LocalKikimr;
import ru.yandex.solomon.kikimr.YdbHelper;
import ru.yandex.solomon.project.manager.abc.db.ydb.YdbAbcToProjectMappingDao;
import ru.yandex.solomon.project.manager.api.v3.intranet.impl.AbcToProjectMappingServiceImpl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class AbcToProjectMappingServiceTest {

    private AbcToProjectMappingService service;
    @ClassRule
    public static LocalKikimr localKikimr = new LocalKikimr();
    @Rule
    public TestName name = new TestName();

    private YdbHelper ydb;
    private YdbProjectsDao projectsDao;
    private YdbAbcToProjectMappingDao dao;

    @Before
    public void setUp() {
        ydb = new YdbHelper(localKikimr, getClass().getSimpleName() + "_" + name.getMethodName());
        projectsDao = new YdbProjectsDao(ydb.getTableClient(), ydb.resolvePath("Projects"), new ObjectMapper(), ForkJoinPool.commonPool());
        dao = new YdbAbcToProjectMappingDao(ydb.getRootPath(), ydb.getTableClient(), ydb.getSchemeClient());
        join(dao.createSchemaForTests());
        join(projectsDao.createSchemaForTests());
        service = new AbcToProjectMappingServiceImpl(
                dao,
                projectsDao,
                Authorizer.anonymous()
        );
    }

    @After
    public void tearDown() {
        join(dao.dropSchemaForTests());
        join(projectsDao.dropSchemaForTests());
        ydb.close();
    }

    @Test
    public void get_failed() {
        projectsDao.insert(ru.yandex.solomon.core.db.model.Project.newBuilder()
                .setId("solomon")
                .setName("new")
                .setOwner("my")
                .setAbcService("abc")
                .build())
                .join();
        projectsDao.insert(ru.yandex.solomon.core.db.model.Project.newBuilder()
                .setId("solomon2")
                .setName("new")
                .setOwner("my")
                .setAbcService("abc")
                .build())
                .join();
        Project result = null;
        try {
            result = service.getDefaultProjectForAbc(GetAbcDefaultProjectRequest.newBuilder()
                    .setAbcService("abc_no_project")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "no single project for abc 'abc_no_project'");
        }
        assertNull(result);

        try {
            result = service.getDefaultProjectForAbc(GetAbcDefaultProjectRequest.newBuilder()
                    .setAbcService("abc")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            NotFoundException ex = (NotFoundException) Throwables.getRootCause(e);
            assertEquals(ex.getMessage(), "no single project for abc 'abc'");
        }
        assertNull(result);
    }

    @Test
    public void get() {
        projectsDao.insert(ru.yandex.solomon.core.db.model.Project.newBuilder()
                .setId("solomon")
                .setName("new")
                .setOwner("my")
                .setAbcService("abc")
                .build())
                .join();
        Project result = service.getDefaultProjectForAbc(GetAbcDefaultProjectRequest.newBuilder()
                    .setAbcService("abc")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(result.getProjectId(), "solomon");
    }

    @Test
    public void update() {
        projectsDao.insert(ru.yandex.solomon.core.db.model.Project.newBuilder()
                .setId("solomon")
                .setName("new")
                .setOwner("my")
                .setAbcService("abc")
                .build())
                .join();
        projectsDao.insert(ru.yandex.solomon.core.db.model.Project.newBuilder()
                .setId("solomon2")
                .setName("new")
                .setOwner("my")
                .setAbcService("abc")
                .build())
                .join();
        service.setDefaultProjectForAbc(SetAbcDefaultProjectRequest.newBuilder()
                .setAbcService("abc")
                .setProjectId("solomon2")
                .build(), AnonymousAuthSubject.INSTANCE).join();
        Project result = service.getDefaultProjectForAbc(GetAbcDefaultProjectRequest.newBuilder()
                .setAbcService("abc")
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(result.getProjectId(), "solomon2");
    }
}
