package ru.yandex.solomon.project.manager.api.v3.intranet;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ForkJoinPool;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Throwables;
import com.google.protobuf.Empty;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monitoring.api.v3.project.manager.CreateProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.DeleteProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.GetProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectsRequest;
import ru.yandex.monitoring.api.v3.project.manager.Project;
import ru.yandex.monitoring.api.v3.project.manager.Role;
import ru.yandex.monitoring.api.v3.project.manager.UpdateProjectRequest;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.abc.validator.AbcServiceFieldValidator;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.config.protobuf.project.manager.ProjectsConfig;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.kikimr.LocalKikimr;
import ru.yandex.solomon.kikimr.YdbHelper;
import ru.yandex.solomon.project.manager.api.v3.intranet.impl.ProjectServiceImpl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ProjectServiceTest {

    private ProjectServiceImpl service;
    @ClassRule
    public static LocalKikimr localKikimr = new LocalKikimr();
    @Rule
    public TestName name = new TestName();

    private YdbHelper ydb;
    private YdbProjectsDao dao;
    private StubListener listener;

    @Before
    public void setUp() {
        ydb = new YdbHelper(localKikimr, getClass().getSimpleName() + "_" + name.getMethodName());
        dao = new YdbProjectsDao(ydb.getTableClient(), ydb.resolvePath("Projects"), new ObjectMapper(), ForkJoinPool.commonPool());
        join(dao.createSchemaForTests());
        service = new ProjectServiceImpl(
                Authorizer.anonymous(),
                dao,
                new AbcServiceFieldValidator(Optional.empty()),
                List.of(listener = new StubListener()),
                new MetricRegistry(),
                new FeatureFlagHolderStub(),
                Optional.of(false),
                Optional.of(ProjectsConfig.getDefaultInstance())
        );
    }

    @After
    public void tearDown() {
        join(dao.dropSchemaForTests());
        ydb.close();
    }

    @Test
    public void get_failed() {
        Project result = null;
        try {
            result = service.get(GetProjectRequest.newBuilder().build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "project ID cannot be blank");
        }
        assertNull(result);

        try {
            result = service.get(GetProjectRequest.newBuilder()
                    .setProjectId("1")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            NotFoundException ex = (NotFoundException) Throwables.getRootCause(e);
            assertEquals(ex.getMessage(), "no project with id '1'");
        }
        assertNull(result);
    }

    @Test
    public void get() {
        var p1 = createRequest();
        var resultCreate = service.create(p1, AnonymousAuthSubject.INSTANCE).join();
        assertCreate(p1, resultCreate);

        var resultGet = service.get(GetProjectRequest.newBuilder()
                .setProjectId(p1.getProjectId())
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertCreate(p1, resultGet);
        assertEquals(resultCreate, resultGet);
    }

    @Test
    public void create_failed() {
        Project result = null;
        try {
            result = service.create(CreateProjectRequest.newBuilder().build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "ID_NOT_SET not supported");
        }
        assertNull(result);
        try {
            result = service.create(CreateProjectRequest.newBuilder()
                    .setProjectId(" ")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "project id cannot be blank");
        }
        assertNull(result);
        try {
            result = service.create(CreateProjectRequest.newBuilder()
                    .setProjectId("id")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "name cannot be blank");
        }
        assertNull(result);
        try {
            result = service.create(CreateProjectRequest.newBuilder()
                    .setProjectId("id")
                    .setName("name")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "abcService cannot be blank");
        }
        assertNull(result);
    }

    @Test
    public void create() {
        var request1 = createRequest();
        var p1 = service.create(request1, AnonymousAuthSubject.INSTANCE).join();
        assertCreate(request1, p1);
    }

    @Test
    public void update_failed() {
        Project result = null;
        try {
            result = service.update(UpdateProjectRequest.newBuilder().build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "ID_NOT_SET not supported");
        }
        assertNull(result);
        try {
            result = service.update(UpdateProjectRequest.newBuilder()
                    .setProjectId(" ")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "project id cannot be blank");
        }
        assertNull(result);
        try {
            result = service.update(UpdateProjectRequest.newBuilder()
                    .setProjectId("id")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "name cannot be blank");
        }
        assertNull(result);
        try {
            result = service.update(UpdateProjectRequest.newBuilder()
                    .setProjectId("id")
                    .setName("name")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "abcService cannot be blank");
        }
        assertNull(result);
    }

    @Test
    public void update() {
        var p = projectEntity();
        dao.insert(p).join();
        var update = UpdateProjectRequest.newBuilder()
                .setProjectId(p.getId())
                .setName(UUID.randomUUID().toString())
                .setDescription(UUID.randomUUID().toString())
                .setAbcService(UUID.randomUUID().toString())
                .putAllLabels(Map.of(UUID.randomUUID().toString(), UUID.randomUUID().toString()))
                .setEtag(p.getVersion() + "")
                .build();
        var updated = service.update(update, AnonymousAuthSubject.INSTANCE).join();

        var resultDb = dao.findById(p.getId()).join().get();

        assertEquals(updated.getProjectId(), update.getProjectId());
        assertEquals(updated.getName(), update.getName());
        assertEquals(updated.getDescription(), update.getDescription());
        assertEquals(updated.getLabelsMap(), update.getLabelsMap());
        assertEquals(updated.getAbcService(), update.getAbcService());
        assertEquals("1", updated.getEtag());

        Assert.assertEquals(resultDb.getName(), update.getName());
        Assert.assertEquals(resultDb.getDescription(), update.getDescription());
        Assert.assertEquals(resultDb.getLabels(), update.getLabelsMap());
        Assert.assertEquals(resultDb.getAbcService(), update.getAbcService());

        Assert.assertEquals(resultDb.getOwner(), p.getOwner());
        Assert.assertEquals(resultDb.getAcl(), p.getAcl());
        Assert.assertEquals(resultDb.isOnlyAuthPush(), p.isOnlyAuthPush());
        Assert.assertEquals(resultDb.isOnlyAuthRead(), p.isOnlyAuthRead());
        Assert.assertEquals(resultDb.isOnlyNewFormatReads(), p.isOnlyNewFormatReads());
        Assert.assertEquals(resultDb.isOnlyNewFormatWrites(), p.isOnlyNewFormatWrites());
        Assert.assertEquals(resultDb.isOnlyMetricNameShards(), p.isOnlyMetricNameShards());
        Assert.assertEquals(resultDb.getMetricNameLabel(), p.getMetricNameLabel());
    }

    @Test
    public void delete() {
        var request1 = createRequest();
        var p1 = service.create(request1, AnonymousAuthSubject.INSTANCE).join();
        service.delete(DeleteProjectRequest.newBuilder().setProjectId(p1.getProjectId()).build(), AnonymousAuthSubject.INSTANCE).join();

        try {
            var resultGet = service.get(GetProjectRequest.newBuilder()
                    .setProjectId(p1.getProjectId())
                    .build(), AnonymousAuthSubject.INSTANCE).join();
            assertNull(resultGet);
        } catch (CompletionException e) {
            NotFoundException ex = (NotFoundException) Throwables.getRootCause(e);
            assertEquals(ex.getMessage(), "no project with id '" + p1.getProjectId() + "'");
        }
    }

    @Test
    public void delete_failed() {
        Empty result = null;
        try {
            result = service.delete(DeleteProjectRequest.newBuilder().build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException e) {
            assertEquals(Throwables.getRootCause(e).getMessage(), "project ID cannot be blank");
        }
        assertNull(result);
    }

    @Test
    public void list() {
        var request1 = createRequest();
        var request2 = createRequest();
        var request3 = createRequest();
        var request4 = createRequest();
        var request5 = createRequest();
        var p1 = service.create(request1, AnonymousAuthSubject.INSTANCE).join();
        var p2 = service.create(request2, AnonymousAuthSubject.INSTANCE).join();
        var p3 = service.create(request3, AnonymousAuthSubject.INSTANCE).join();
        var p4 = service.create(request4, AnonymousAuthSubject.INSTANCE).join();
        var p5 = service.create(request5, AnonymousAuthSubject.INSTANCE).join();

        var result = service.list(ListProjectsRequest.newBuilder()
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals("", result.getNextPageToken());
        assertTrue(result.getProjectsList().contains(p1));
        assertTrue(result.getProjectsList().contains(p2));
        assertTrue(result.getProjectsList().contains(p3));
        assertTrue(result.getProjectsList().contains(p4));
        assertTrue(result.getProjectsList().contains(p5));
        assertEquals(result.getProjectsCount(), 5);

        result = service.list(ListProjectsRequest.newBuilder()
                .setPageSize(2)
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertNotEquals("", result.getNextPageToken());
        assertEquals(result.getProjectsCount(), 2);
        result = service.list(ListProjectsRequest.newBuilder()
                .setPageSize(2)
                .setPageToken(result.getNextPageToken())
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertNotEquals("", result.getNextPageToken());
        assertEquals(result.getProjectsCount(), 2);
        result = service.list(ListProjectsRequest.newBuilder()
                .setPageSize(2)
                .setPageToken(result.getNextPageToken())
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals("", result.getNextPageToken());
        assertEquals(result.getProjectsCount(), 1);

        result = service.list(ListProjectsRequest.newBuilder()
                .addFilterByRole(Role.ROLE_PROJECT_ADMIN)
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals("", result.getNextPageToken());
        assertEquals(result.getProjectsCount(), 5);
    }

    @Test
    public void listener() {
        var request1 = createRequest();
        service.create(request1, AnonymousAuthSubject.INSTANCE).join();
        var createdProject = dao.findById(request1.getProjectId()).join().get();

        assertEquals(listener.createdProject, createdProject);

        var update = UpdateProjectRequest.newBuilder()
                .setProjectId(createdProject.getId())
                .setName(UUID.randomUUID().toString())
                .setDescription(UUID.randomUUID().toString())
                .setAbcService(UUID.randomUUID().toString())
                .putAllLabels(Map.of(UUID.randomUUID().toString(), UUID.randomUUID().toString()))
                .setEtag(createdProject.getVersion() + "")
                .build();
        service.update(update, AnonymousAuthSubject.INSTANCE).join();
        var updatedProject = dao.findById(request1.getProjectId()).join().get();

        assertEquals(listener.updatedProject, updatedProject);

        var delete = DeleteProjectRequest.newBuilder().setProjectId(request1.getProjectId()).build();
        service.delete(delete, AnonymousAuthSubject.INSTANCE).join();
        assertEquals(listener.preDeleteActionRequest, delete);
        assertEquals(listener.deleteRequest, delete);
    }

    private void assertCreate(CreateProjectRequest p1, Project result) {
        assertEquals(p1.getProjectId(), result.getProjectId());
        assertEquals(p1.getName(), result.getName());
        assertEquals(p1.getDescription(), result.getDescription());
        assertEquals(p1.getLabelsMap(), result.getLabelsMap());
        assertEquals(p1.getAbcService(), result.getAbcService());
        assertEquals("0", result.getEtag());
    }

    private CreateProjectRequest createRequest() {
        return CreateProjectRequest.newBuilder()
                .setProjectId(UUID.randomUUID().toString())
                .setName(UUID.randomUUID().toString())
                .setDescription(UUID.randomUUID().toString())
                .setAbcService(UUID.randomUUID().toString())
                .putAllLabels(Map.of(UUID.randomUUID().toString(), UUID.randomUUID().toString()))
                .build();
    }

    private ru.yandex.solomon.core.db.model.Project projectEntity() {
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        var id = UUID.randomUUID().toString();
        return new ru.yandex.solomon.core.db.model.Project(
                id, StringUtils.capitalize(id), "Description for " + id, UUID.randomUUID().toString(),
                Acl.of(Set.of(UUID.randomUUID().toString()), Set.of(UUID.randomUUID().toString()), Set.of(UUID.randomUUID().toString()), Set.of(UUID.randomUUID().toString())), id,
                true, false, true, true,
                true, "sensor",
                now, now, UUID.randomUUID().toString(), UUID.randomUUID().toString(), 0,
                Map.of("label1", "value1", "label2", "value2")
        );
    }

    private class StubListener implements ProjectChangeListener {

        private ru.yandex.solomon.core.db.model.Project createdProject;
        private ru.yandex.solomon.core.db.model.Project updatedProject;
        private DeleteProjectRequest deleteRequest;
        private DeleteProjectRequest preDeleteActionRequest;

        @Override
        public CompletableFuture<Void> create(ru.yandex.solomon.core.db.model.Project result) {
            this.createdProject = result;
            return CompletableFuture.completedFuture(null);
        }

        @Override
        public CompletableFuture<Void> update(ru.yandex.solomon.core.db.model.Project result) {
            this.updatedProject = result;
            return CompletableFuture.completedFuture(null);
        }

        @Override
        public CompletableFuture<Void> delete(DeleteProjectRequest request) {
            this.deleteRequest = request;
            return CompletableFuture.completedFuture(null);
        }

        @Override
        public CompletableFuture<Void> preDeleteAction(DeleteProjectRequest request) {
            this.preDeleteActionRequest = request;
            return CompletableFuture.completedFuture(null);
        }
    }
}
