package ru.yandex.solomon.project.manager.api.v3.intranet.dto;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monitoring.api.v3.project.manager.CreateProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.UpdateProjectRequest;
import ru.yandex.solomon.config.protobuf.project.manager.ProjectsConfig;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ProjectDtoConverterTest {

    private ProjectDtoConverter converter = new ProjectDtoConverter();

    @Test
    public void fromEntity() {
        var p = projectEntity();
        var result = converter.fromEntity(p);
        assertProject(p, result);
    }

    @Test
    public void toEntity() {
        var create = CreateProjectRequest.newBuilder()
                .setProjectId(UUID.randomUUID().toString())
                .setName(UUID.randomUUID().toString())
                .setDescription(UUID.randomUUID().toString())
                .setAbcService(UUID.randomUUID().toString())
                .putAllLabels(Map.of(UUID.randomUUID().toString(), UUID.randomUUID().toString()))
                .build();
        var p = converter.toEntity(create, "123", ProjectsConfig.getDefaultInstance());
        assertEquals("123", p.getCreatedBy());
        assertEquals("", p.getUpdatedBy());
        assertEquals(create.getProjectId(), p.getId());
        assertEquals(create.getName(), p.getName());
        assertEquals(create.getDescription(), p.getDescription());
        assertEquals(create.getAbcService(), p.getAbcService());
        assertEquals(create.getLabelsMap(), p.getLabels());
        Assert.assertEquals("123", p.getOwner());
        Assert.assertEquals(Acl.empty(), p.getAcl());
        Assert.assertEquals(true, p.isOnlyAuthPush());
        Assert.assertEquals(false, p.isOnlyAuthRead());
        Assert.assertEquals(false, p.isOnlyNewFormatReads());
        Assert.assertEquals(false, p.isOnlyNewFormatWrites());
        Assert.assertEquals(false, p.isOnlyMetricNameShards());
        Assert.assertEquals("", p.getMetricNameLabel());
        assertEquals(0, p.getVersion());
    }

    @Test
    public void toEntityUpdate() {
        var p = projectEntity();
        var updateProjectRequest = UpdateProjectRequest.newBuilder()
                .setProjectId(UUID.randomUUID().toString())
                .setName(UUID.randomUUID().toString())
                .setDescription(UUID.randomUUID().toString())
                .setAbcService(UUID.randomUUID().toString())
                .putAllLabels(Map.of(UUID.randomUUID().toString(), UUID.randomUUID().toString()))
                .setEtag("123")
                .build();
        var result = converter.toEntity(updateProjectRequest, "123", p);
        assertEquals("123", result.getUpdatedBy());
        assertEquals(p.getCreatedBy(), result.getCreatedBy());
        assertEquals(updateProjectRequest.getProjectId(), result.getId());
        assertEquals(updateProjectRequest.getName(), result.getName());
        assertEquals(updateProjectRequest.getDescription(), result.getDescription());
        assertEquals(updateProjectRequest.getAbcService(), result.getAbcService());
        assertEquals(updateProjectRequest.getLabelsMap(), result.getLabels());
        assertEquals(updateProjectRequest.getEtag(), String.valueOf(result.getVersion()));
        Assert.assertEquals(result.getOwner(), p.getOwner());
        Assert.assertEquals(result.getAcl(), p.getAcl());
        Assert.assertEquals(result.isOnlyAuthPush(), p.isOnlyAuthPush());
        Assert.assertEquals(result.isOnlyAuthRead(), p.isOnlyAuthRead());
        Assert.assertEquals(result.isOnlyNewFormatReads(), p.isOnlyNewFormatReads());
        Assert.assertEquals(result.isOnlyNewFormatWrites(), p.isOnlyNewFormatWrites());
        Assert.assertEquals(result.isOnlyMetricNameShards(), p.isOnlyMetricNameShards());
        Assert.assertEquals(result.getMetricNameLabel(), p.getMetricNameLabel());
        Assert.assertEquals(result.getOwner(), p.getOwner());
    }

    @Test
    public void fromEntityPage() {
        var p = projectEntity();
        var p2 = projectEntity();
        var result = converter.fromEntity(PagedResult.of(List.of(p, p2), new PageOptions(2, 1), 2));
        assertEquals(2, result.getProjectsCount());
        assertEquals("", result.getNextPageToken());

        result = converter.fromEntity(PagedResult.of(List.of(p, p2), new PageOptions(2, 1), 22));
        assertEquals(2, result.getProjectsCount());
        assertNotEquals("", result.getNextPageToken());
    }

    private void assertProject(Project p, ru.yandex.monitoring.api.v3.project.manager.Project result) {
        assertEquals(result.getProjectId(), p.getId());
        assertEquals(result.getName(), p.getName());
        assertEquals(result.getDescription(), p.getDescription());
        assertEquals(result.getCreatedBy(), p.getCreatedBy());
        assertEquals(result.getModifiedBy(), p.getUpdatedBy());
        assertEquals(result.getAbcService(), p.getAbcService());
        assertEquals(result.getLabelsMap(), p.getLabels());
        assertEquals(result.getEtag(), String.valueOf(p.getVersion()));
        assertEquals(Timestamps.toMillis(result.getCreatedAt()), p.getCreatedAtMillis());
        assertEquals(Timestamps.toMillis(result.getModifiedAt()), p.getUpdatedAtMillis());
    }

    private Project projectEntity() {
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        var id =  UUID.randomUUID().toString();
        return new Project(
                id, StringUtils.capitalize(id), "Description for " + id, UUID.randomUUID().toString(),
                Acl.of(Set.of(UUID.randomUUID().toString()), Set.of(UUID.randomUUID().toString()), Set.of(UUID.randomUUID().toString()), Set.of(UUID.randomUUID().toString())), id,
                true, false, true, true,
                true, "sensor",
                now, now, UUID.randomUUID().toString(), UUID.randomUUID().toString(), 0,
                Map.of("label1", "value1", "label2", "value2")
        );
    }
}
