#include <solomon/libs/cpp/actors/metrics/actor_runtime_metrics.h>
#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <solomon/libs/cpp/cluster_map/cluster.h>
#include <solomon/libs/cpp/cluster_membership/actor.h>
#include <solomon/libs/cpp/config_includes/config_includes.h>
#include <solomon/libs/cpp/grpc/executor/limits.h>
#include <solomon/libs/cpp/grpc/server/server.h>
#include <solomon/libs/cpp/http/server/core/http_server.h>
#include <solomon/libs/cpp/http/server/handlers/metrics.h>
#include <solomon/libs/cpp/http/server/handlers/version.h>
#include <solomon/libs/cpp/minidump/minidump.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/selfmon/service/proto_page.h>
#include <solomon/libs/cpp/selfmon/service/service.h>
#include <solomon/libs/cpp/signals/signals.h>
#include <solomon/libs/cpp/clients/slicer/slicer_client.h>
#include <solomon/libs/cpp/ydb/driver.h>

#include <solomon/services/slicer/lib/api/service.h>
#include <solomon/services/slicer/lib/assignment_manager/service_manager.h>
#include <solomon/services/slicer/lib/config/config.h>
#include <solomon/services/slicer/lib/db/assignment_dao.h>
#include <solomon/services/slicer/lib/http_server/handlers.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/svnversion/svnversion.h>

#include <util/folder/path.h>
#include <util/system/getpid.h>

using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon::NSlicerClient;
using namespace NSolomon::NSlicer::NApi;
using namespace NSolomon::NSlicer;
using namespace NSolomon::NSlicer::NDb;
using namespace NSolomon;
using yandex::monitoring::slicer::SlicerConfig;
using yandex::solomon::config::http::HttpServerConfig;


void ResolveAddressesInGrpcClientConfig(yandex::solomon::config::rpc::TGrpcClientConfig& conf) {
    TStringBuilder sb;
    for (const auto& addr: conf.GetAddresses()) {
        sb << addr << '\n';
    }

    auto cluster = TClusterMapBase::Load(sb);
    conf.ClearAddresses();

    for (auto&& node: cluster->Nodes()) {
        conf.AddAddresses(node.Endpoint);
    }
}

ISlicerClusterClientPtr CreateSlicerClients(SlicerConfig& config, NMonitoring::TMetricRegistry& registry) {
    auto* grpcConfig = config.mutable_slicer_client_config();

    ResolveAddressesInGrpcClientConfig(*grpcConfig);

    Cerr << "Slicer addresses resolved into: " << Endl;
    for (const auto& addr: grpcConfig->addresses()) {
        Cerr << "  " << addr << Endl;
    }
    Cerr << Endl;

    return NSlicerClient::CreateSlicerGrpcClusterClient(
            *grpcConfig,
            registry,
            config.client_id());
}

int Main(SlicerConfig& config, const std::shared_ptr<NSecrets::ISecretProvider>& secretProvider) {
    NSolomon::NGrpc::SetThreadsLimit(2);
    auto registry = std::make_shared<NMonitoring::TMetricRegistry>();

    // actor system
    auto actorRuntime = TActorRuntime::Create(config.actor_system(), registry);
    actorRuntime->Start();

    Y_ENSURE(config.has_ydb_config(), "ydb_config must be present");
    auto ydbDriver = CreateDriver(config.ydb_config(), *secretProvider);

    Y_ENSURE(!config.table_path_prefix().empty(), "table path prefix must be present");
    auto client = std::make_shared<NYdb::NTable::TTableClient>(ydbDriver);
    auto assignmentDao = CreateYdbAssignmentDao(config.table_path_prefix() + "assignments", client, *registry);
    auto serviceDao = CreateYdbServiceConfigDao(config.table_path_prefix() + "ServiceConfigs", std::move(client), *registry);

    Y_ENSURE(!config.dc().empty(), "dc must be present");
    Y_ENSURE(!config.cluster().empty(), "cluster must be present");

    Y_ENSURE(config.has_leader_election_config(), "leader election config must be present");
    Y_ENSURE(config.leader_election_config().has_lock_config(), "lock config must be present");

    auto clusterMembershipFactory = [](
            const TString& service,
            const SlicerConfig& config,
            NMonitoring::TMetricRegistry& registry)
    {
        return NSolomon::NClusterMembership::CreateClusterMembershipActor(
            service,
            config.cluster_membership(),
            config.api_server().GetPort(0),
            registry,
            {},
            config.client_id());
    };

    auto slicerClients = CreateSlicerClients(config, *registry);

    auto serviceManagerActor = CreateServiceManager(
            // TODO(ivanzhukov@): support a grpc config from the root config
            config,
            ydbDriver,
            std::move(clusterMembershipFactory),
            std::move(slicerClients),
            *registry,
            config.client_id(),
            std::move(assignmentDao),
            std::move(serviceDao));
    TActorId serviceManager = actorRuntime->Register(
            serviceManagerActor.release(),
            NActors::TMailboxType::HTSwap,
            actorRuntime->FindExecutorByName("assignments")); // TODO: get a name from the config

    AddActorRuntimeMetrics(*actorRuntime, *registry);
    registry->IntGauge({{"sensor", "version"}, {"revision", GetArcadiaLastChange()}})->Set(1);

    std::unique_ptr<NSolomon::NHttp::THttpServer> httpServer;
    if (config.has_http_server()) {
        httpServer = std::make_unique<NSolomon::NHttp::THttpServer>(*actorRuntime, registry, config.http_server());
        httpServer->Handle("/metrics", NSolomon::NHttp::CreateMetricsHandler(registry));
        httpServer->Handle("/version", NSolomon::NHttp::TVersionHandler{});

        NSelfMon::InitService(registry, actorRuntime->ActorSystem(), httpServer->Proxy(), httpServer->ExecutorId());
        InitSlicerSpecificHttpHandlers(actorRuntime->ActorSystem(), serviceManager);

        NSelfMon::RegisterPage(
                actorRuntime->ActorSystem(),
                "/config", "Config",
                NSelfMon::ProtoPage(std::make_shared<SlicerConfig>(config)));
    }

    // gRPC server
    auto apiServer = NSolomon::NGrpc::MakeGRpcServerFromConfig(actorRuntime->ActorSystem(), *registry, config.api_server());
    {
        ui32 executorPool = actorRuntime->FindExecutorByName(config.api_server().GetThreadPoolName());
        apiServer->AddService(MakeIntrusive<NApi::TSlicerService>(
            actorRuntime->ActorSystem(), *registry, executorPool, serviceManager));
        apiServer->Start();
    }

    // main loop
    sigset_t blockMask;
    SigEmptySet(&blockMask);

    while (true) {
        SigSuspend(&blockMask);

        if (NeedTerminate) {
            actorRuntime->EmergencyLog("Slicer was terminated");
            apiServer->Stop();

            // TODO: stop the http server gracefully

            // TODO(ivanzhukov@):
//            auto f = NThreading::WaitAll(
//                actorRuntime->AsyncPoison(assignmentManagerId));
//            f.GetValueSync();

            apiServer = nullptr;
            actorRuntime = nullptr;
            return 0;
        } else if (NeedReopenLog) {
            NeedReopenLog = 0;
            actorRuntime->ReopenLog();
            TLogBackend::ReopenAllBackends();
        }
    }
}

int main(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts;
    opts.AddLongOption("config", "path to configuration file")
            .Required()
            .RequiredArgument("PATH");
    opts.AddLongOption("secrets", "path to secrets file")
            .Required()
            .RequiredArgument("<path>")
            .Completer(NLastGetopt::NComp::File("*.secrets"));
    opts.AddVersionOption();
    opts.AddHelpOption();
    opts.SetFreeArgsNum(0);

    try {
        NLastGetopt::TOptsParseResult r(&opts, argc, argv);

        InitSignals();
        Cerr << "Starting Slicer (pid: " << GetPID() << ", SVN revision: " << GetArcadiaLastChange() << ')' << Endl;

        auto config = ParseTextProto(r.Get("config"));
        MergeIncludes(config);
        Validate(config);

        SetupMinidump(config.minidump_config());

        TFsPath secrets(r.Get("secrets"));
        auto secretProvider = secrets.Exists()
            ? NSecrets::FileSecretProvider(secrets.GetPath())
            : NSecrets::EmptySecretProvider();

        return Main(config, secretProvider);
    } catch (const NLastGetopt::TUsageException&) {
        opts.PrintUsage("slicer");
    } catch (const TConfigParseException& e) {
        Cerr << "Cannot parse config: " << e.AsStrBuf() << Endl;
    } catch (const TConfigValidationException& e) {
        Cerr << "Config is not valid: " << e.AsStrBuf() << Endl;
    } catch (...) {
        Cerr << "Unhandled exception: " << CurrentExceptionMessage() << ". Process will terminate" << Endl;
    }

    return 1;
}
