#include "assignments.h"

using namespace yandex::monitoring::slicer;

namespace NSolomon::NSlicer::NApi {

TGetAllAssignmentsHandler::TGetAllAssignmentsHandler(NActors::TActorId serviceManager)
    : ServiceManager_{serviceManager}
{
}

TGetSlicesByHostHandler::TGetSlicesByHostHandler(NActors::TActorId serviceManager)
    : ServiceManager_{serviceManager}
{
}

void TGetAllAssignmentsHandler::Handle(const NActors::TActorContext&) {
    auto* request = GetRequest();

    const auto& service = request->service();
    if (service.empty()) {
        GetRequestCtx()->ReplyError(grpc::INVALID_ARGUMENT, "service cannot be empty");
        PassAway();
        return;
    }

    auto ev = std::make_unique<TAssignmentEvents::TGetAllAssignments>();
    ev->Service = service;

    Send(ServiceManager_, ev.release());
}

void TGetAllAssignmentsHandler::OnAssignments(TAssignmentEvents::TGetAllAssignmentsResult::TPtr& ev) {
    auto& assignments = ev->Get()->Assignments;

    auto* arena = GetRequestCtx()->GetArena();
    auto* reply = google::protobuf::Arena::CreateMessage<GetAllAssignmentsResponse>(arena);

    for (const auto& [slice, hosts]: assignments) {
        auto* protoAssignment = reply->add_assignments();
        protoAssignment->set_slice_start(slice.Start);
        protoAssignment->set_slice_end(slice.End);

        for (const auto& host: hosts) {
            protoAssignment->add_hosts(host);
        }
    }

    reply->set_is_leader(true);

    GetRequestCtx()->Reply(reply);
    PassAway();
}

void TGetAllAssignmentsHandler::OnForwardedAssignments(TAssignmentEvents::TGetAllAssignmentsForwardedResponse::TPtr& ev) {
    auto& resOrErr = ev->Get()->Response;
    auto& leaderAddress = ev->Get()->LeaderAddress;

    GetAllAssignmentsResponse* reply;

    if (!resOrErr) {
        auto* arena = GetRequestCtx()->GetArena();
        reply = google::protobuf::Arena::CreateMessage<GetAllAssignmentsResponse>(arena);
    } else if (resOrErr->Fail()) {
        const auto& err = resOrErr->Error();

        if (err.InternalError) {
            GetRequestCtx()->ReplyError(grpc::StatusCode::INTERNAL, err.Msg);
        } else {
            GetRequestCtx()->ReplyError(static_cast<grpc::StatusCode>(err.GRpcStatusCode), err.Msg);
        }

        PassAway();
        return;
    } else {
        reply = &resOrErr->Value();
    }

    reply->set_is_leader(false);
    reply->set_leader_address(leaderAddress);

    GetRequestCtx()->Reply(reply);
    PassAway();
}

void TGetSlicesByHostHandler::Handle(const NActors::TActorContext&) {
    auto* request = GetRequest();

    const auto& service = request->service();
    if (service.empty()) {
        GetRequestCtx()->ReplyError(grpc::INVALID_ARGUMENT, "service cannot be empty");
        PassAway();
        return;
    }

    Send(ServiceManager_, new TAssignmentEvents::TGetSlicesByHost{service, request->Gethost()});
}

void TGetSlicesByHostHandler::OnSlices(TAssignmentEvents::TGetSlicesByHostResult::TPtr& ev) {
    auto& resOrErr = ev->Get()->SlicesResult;

    if (resOrErr.Fail()) {
        const auto& err = resOrErr.Error();

        if (err.InternalError) {
            GetRequestCtx()->ReplyError(grpc::StatusCode::INTERNAL, err.Msg);
        } else {
            GetRequestCtx()->ReplyError(static_cast<grpc::StatusCode>(err.GRpcStatusCode), err.Msg);
        }

        PassAway();
        return;
    }

    auto* arena = GetRequestCtx()->GetArena();
    auto* reply = google::protobuf::Arena::CreateMessage<GetSlicesByHostResponse>(arena);

    auto& result = resOrErr.Value();

    // TODO(ivanzhukov): send only diff -- assigned + unassigned slices
    for (const auto& slice: result.Slices) {
        reply->add_assigned_starts(slice.Start);
        reply->add_assigned_ends(slice.End);
    }

    reply->set_is_leader(true);

    GetRequestCtx()->Reply(reply);
    PassAway();
}


void TGetSlicesByHostHandler::OnForwardedSlices(TAssignmentEvents::TGetSlicesByHostForwardedResponse::TPtr& ev) {
    auto& resOrErr = ev->Get()->Response;
    auto& leaderAddress = ev->Get()->LeaderAddress;

    GetSlicesByHostResponse* reply;

    if (!resOrErr) {
        auto* arena = GetRequestCtx()->GetArena();
        reply = google::protobuf::Arena::CreateMessage<GetSlicesByHostResponse>(arena);
    } else if (resOrErr->Fail()) {
        const auto& err = resOrErr->Error();

        // XXX(ivanzhukov): instead of propagating an error, tell the client who is a leader
        if (err.InternalError) {
            GetRequestCtx()->ReplyError(grpc::StatusCode::INTERNAL, err.Msg);
        } else {
            GetRequestCtx()->ReplyError(static_cast<grpc::StatusCode>(err.GRpcStatusCode), err.Msg);
        }

        PassAway();
        return;
    } else {
        reply = &resOrErr->Value();
    }

    reply->set_is_leader(false);
    reply->set_leader_address(leaderAddress);

    GetRequestCtx()->Reply(reply);
    PassAway();
}

TMoveShardHandler::TMoveShardHandler(NActors::TActorId serviceManager)
    : ServiceManager_{serviceManager}
{}

void TMoveShardHandler::Handle(const NActors::TActorContext&) {
    auto* request = GetRequest();

    const auto& service = request->service();
    if (service.empty()) {
        GetRequestCtx()->ReplyError(grpc::INVALID_ARGUMENT, "service cannot be empty");
        PassAway();
        return;
    }

    Send(ServiceManager_, new TAssignmentEvents::TMoveShardRequest{
        service,
        static_cast<TNumId>(request->num_id()),
        request->host()});
}

void TMoveShardHandler::OnMoveShardResponse(TAssignmentEvents::TMoveShardResponse::TPtr& ev) {
    const auto& status = ev->Get()->Status;

    if (!status.Ok()) {
        if (status.InternalError) {
            GetRequestCtx()->ReplyError(grpc::StatusCode::INTERNAL, status.Msg);
        } else {
            GetRequestCtx()->ReplyError(static_cast<grpc::StatusCode>(status.GRpcStatusCode), status.Msg);
        }
    } else {
        auto* arena = GetRequestCtx()->GetArena();
        auto* reply = google::protobuf::Arena::CreateMessage<MoveShardResponse>(arena);
        reply->set_is_ok(true);

        GetRequestCtx()->Reply(reply);
    }

    PassAway();
}

} // namespace NSolomon::NSlicer::NApi
