#include "assignment_manager.h"
#include "events.h"
#include "private_events.h"
#include "request_ctx_fwd_get_all_assn.h"
#include "request_ctx_fwd_get_slices_by_host.h"
#include "request_ctx_load_info.h"

#include <solomon/services/slicer/lib/balancer/balancer.h>
#include <solomon/services/slicer/lib/common/assignments.h>
#include <solomon/services/slicer/lib/common/host_reported.h>
#include <solomon/services/slicer/lib/common/reassignment_type.h>
#include <solomon/services/slicer/lib/db/service_config.h>

#include <solomon/libs/cpp/actors/util/requests_executor.h>
#include <solomon/libs/cpp/clients/load_info/load_service_client.h>
#include <solomon/libs/cpp/cluster_membership/events.h>
#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/coordination/leader_election/leader.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/slices/operations.h>

#include <library/cpp/actors/core/events.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>

#include <util/generic/ptr.h>
#include <util/generic/scope.h>
#include <util/generic/set.h>

#include <memory>

using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon::NClusterMembership;
using namespace NSolomon::NCoordination;
using namespace NSolomon::NLoadInfo;
using namespace NSolomon::NSlicerClient;
using namespace yandex::monitoring::slicer;

namespace NSolomon::NSlicer::NApi {
namespace {

constexpr TDuration PROBING_INTERVAL_DEFAULT = TDuration::Seconds(15);
constexpr TDuration PROBING_BACKOFF_DEFAULT = TDuration::Seconds(5);
constexpr TDuration PROBING_TIMEOUT_DEFAULT = TDuration::Seconds(30);
constexpr TDuration REASSIGNMENT_FUSE = TDuration::Minutes(5); // SOLOMON-7819. Remove after SOLOMON-7822
constexpr TDuration LEADER_ELECTION_INTERVAL_DEFAULT = TDuration::Seconds(5);

class TMetrics {
public:
    TMetrics(
            const TString& serviceName, // Ingestor/Memstore/etc.
            NMonitoring::TMetricRegistry& registry)
    {
        auto createServiceLabels = CreateServiceLabelsFactory(serviceName);
        ServiceClusterSize = registry.IntGauge(createServiceLabels("clusterSize"));
        ServiceAliveNodes = registry.IntGauge(createServiceLabels("aliveNodes"));
        ServiceSuspiciousNodes = registry.IntGauge(createServiceLabels("suspiciousNodes"));
        ServiceDeadNodes = registry.IntGauge(createServiceLabels("deadNodes"));
        ServiceUnknownNodes = registry.IntGauge(createServiceLabels("unknownNodes"));
        SlicesCount = registry.IntGauge(createServiceLabels("slicesCount"));
        ShardsCount = registry.IntGauge(createServiceLabels("shardsCount"));
        IsLeader = registry.IntGauge(createServiceLabels("isLeader"));
        BalancingTookMs = registry.IntGauge(createServiceLabels("balancing.tookMs"));
        BalancingMoveKeyChurn = registry.Gauge(createServiceLabels("balancing.moveKeyChurn"));
    }

public:
    void Reset() const {
        ServiceClusterSize->Set(0);
        ServiceAliveNodes->Set(0);
        ServiceSuspiciousNodes->Set(0);
        ServiceDeadNodes->Set(0);
        ServiceUnknownNodes->Set(0);
        SlicesCount->Set(0);
        ShardsCount->Set(0);
        IsLeader->Set(0);
        BalancingTookMs->Set(0);
        BalancingMoveKeyChurn->Set(0);
    }

private:
    static std::function<TLabels(TStringBuf)> CreateServiceLabelsFactory(const TString& serviceName) {
        return [serviceName](TStringBuf metricName) {
            auto labels = TLabels{};
            labels.Add("component", serviceName);
            labels.Add("sensor", metricName);

            return labels;
        };
    }

public:
    TIntGauge* ServiceClusterSize;
    TIntGauge* ServiceAliveNodes;
    TIntGauge* ServiceSuspiciousNodes;
    TIntGauge* ServiceDeadNodes;
    TIntGauge* ServiceUnknownNodes;
    /**
     * current number of slices in the system (sum by hosts)
     */
    TIntGauge* SlicesCount;
    TIntGauge* ShardsCount;
    TIntGauge* IsLeader; // 0 or 1
    TIntGauge* BalancingTookMs;
    TGauge* BalancingMoveKeyChurn; // ratio of moved keys, from 0.0 to 1.0
};

enum class EReassignStrategy {
    FORCE,
    IF_NOT_ALREADY,
};

struct TNodesInfo {
    TVector<TStringBuf> DeadNodes;
    TVector<TStringBuf> AssignableNodes;
};

struct THostToSlices {
    TStringBuf Host;
    TSlices Slices;
};

class TAssignmentManager:
        public TActorBootstrapped<TAssignmentManager>,
        public IClusterMember
{
    struct TProxy: IClusterMember {
        TProxy(IClusterMember* impl)
            : Impl_{impl}
        {
        }

        void OnBecomeLeader() override {
            Impl_->OnBecomeLeader();
        }

        void OnBecomeFollower() override {
            Impl_->OnBecomeFollower();
        }

        void OnLeaderChanged(TString leaderId, ui64 orderId) override {
            Impl_->OnLeaderChanged(std::move(leaderId), orderId);
        }

        void OnError(const TString& message) override {
            Impl_->OnError(message);
        }

    private:
        IClusterMember* Impl_{nullptr};
    };

public:
    TAssignmentManager(
            TString serviceName,
            const SlicerConfig& config,
            TActorId clusterMembershipActor,
            IDistributedLockPtr lock,
            ISlicerClusterClientPtr slicerClients,
            NMonitoring::TMetricRegistry& registry,
            TString clientId,
            NDb::IAssignmentDaoPtr assnDao,
            std::shared_ptr<NDb::IServiceConfigDao> serviceDao)
        : Service_{std::move(serviceName)}
        , Config_{config}
        , ProbingOptions_{
                .Interval = FromProtoTime(config.probing().interval(), PROBING_INTERVAL_DEFAULT),
                .MaxRetries = config.probing().max_retries(),
                .Backoff = FromProtoTime(config.probing().backoff(), PROBING_BACKOFF_DEFAULT),
                .Timeout = FromProtoTime(config.probing().timeout(), PROBING_TIMEOUT_DEFAULT),
        }
        , ClusterMembershipActor_{clusterMembershipActor}
        , Lock_{std::move(lock)}
        , LeaderElectionInterval_{FromProtoTime(
                config.leader_election_config().reelection_interval(),
                LEADER_ELECTION_INTERVAL_DEFAULT)
        }
        , SlicerClients_{std::move(slicerClients)}
        , Registry_{registry}
        , Metrics_{Service_, Registry_}
        , LoadInfoClusterClient_{CreateLoadInfoGrpcClusterClient({}, Registry_, std::move(clientId))}
        , ServiceSettings_{}
        , MetricsUpdateInterval_{TDuration::Seconds(5)}
        , AsgnDao_(std::move(assnDao))
        , ServiceDao_{std::move(serviceDao)}
        , Dc_(Config_.dc())
        , Cluster_(Config_.cluster())
    {
// AM - Assignment Membership
#define AM_TRACE(...) MON_TRACE(AssignmentManager, '(' << Service_ << ") " << __VA_ARGS__)
#define AM_DEBUG(...) MON_DEBUG(AssignmentManager, '(' << Service_ << ") " << __VA_ARGS__)
#define AM_INFO(...)  MON_INFO(AssignmentManager,  '(' << Service_ << ") " << __VA_ARGS__)
#define AM_WARN(...)  MON_WARN(AssignmentManager,  '(' << Service_ << ") " << __VA_ARGS__)
#define AM_ERROR(...) MON_ERROR(AssignmentManager, '(' << Service_ << ") " << __VA_ARGS__)
#define AM_CRIT(...)  MON_CRIT(AssignmentManager,  '(' << Service_ << ") " << __VA_ARGS__)

        TStringBuf someNodeAddress = Config_.slicer_client_config().addresses(0);
        TStringBuf tmp;
        someNodeAddress.RSplit(':', tmp, GrpcPort_);
    }

    void Bootstrap() {
        Become(&TThis::FollowerState);

        auto proxy = MakeHolder<TProxy>(this);
        Register(CreateClusterMemberActor(
            Lock_,
            std::move(proxy),
            LeaderElectionInterval_
        ));

        Send(ClusterMembershipActor_, new TGossipEvents::TSubscribe{SelfId()});
        // XXX(ivanzhukov): align time by a grid?
        Schedule(ProbingOptions_.Interval, new TPrivateEvents::TClusterProbing);
        Schedule(MetricsUpdateInterval_, new TPrivateEvents::TMetricsUpdate);
    }

private:
    STATEFN(FollowerState) {
        switch (ev->GetTypeRewrite()) {
            // gRPC
            hFunc(TAssignmentEvents::TGetAllAssignments, OnGetAllAssignmentsRequest);
            hFunc(TAssignmentEvents::TGetSlicesByHost, OnGetSlicesByHostRequest);
            hFunc(TAssignmentEvents::TMoveShardRequest, OnMoveShardRequest);

            // http
            hFunc(TAssignmentEvents::TGetClusterStatus, OnGetClusterStatus);
            hFunc(TAssignmentEvents::TGetShardsHostInfo, OnGetShardsHostInfo);
            hFunc(TAssignmentEvents::TGetHostInfo, OnGetHostInfo);
            hFunc(TAssignmentEvents::TValidateShardMoveParams, OnValidateShardMoveParams);

            // internal
            hFunc(TGossipEvents::TClusterStateResponse, OnClusterStateResponse);
            hFunc(TPrivateEvents::TClusterProbing, OnProbing);
            hFunc(TPrivateEvents::TOneNodeProbingResult, OnOneNodeProbingResult);
            hFunc(TPrivateEvents::TClusterProbingCompleted, OnClusterProbingCompleted);

            hFunc(TPrivateEvents::TLeaderRespondedOnGetAllAssignments, OnLeaderResponseOnGetAllAssignments);
            hFunc(TPrivateEvents::TLeaderRespondedOnGetSlicesByHost, OnLeaderResponseOnGetSlicesByHost);

            sFunc(TPrivateEvents::TMetricsUpdate, OnMetricsUpdate);
        }
    }

    STATEFN(LeaderState) {
        switch (ev->GetTypeRewrite()) {
            // gRPC
            hFunc(TAssignmentEvents::TGetAllAssignments, OnGetAllAssignmentsRequest);
            hFunc(TAssignmentEvents::TGetSlicesByHost, OnGetSlicesByHostRequest);
            hFunc(TAssignmentEvents::TMoveShardRequest, OnMoveShardRequest);

            // http
            hFunc(TAssignmentEvents::TGetClusterStatus, OnGetClusterStatus);
            hFunc(TAssignmentEvents::TGetShardsHostInfo, OnGetShardsHostInfo);
            hFunc(TAssignmentEvents::TGetHostInfo, OnGetHostInfo);
            hFunc(TAssignmentEvents::TValidateShardMoveParams, OnValidateShardMoveParams);
            sFunc(TAssignmentEvents::TAssignUniformly, OnAssignUniformly);
            sFunc(TAssignmentEvents::TRebalanceOnceByCount, OnRebalanceOnceByCount);
            sFunc(TAssignmentEvents::TRebalanceOnceByCpu, OnRebalanceOnceByCpu);
            sFunc(TAssignmentEvents::TRebalanceOnceByMemory, OnRebalanceOnceByMemory);
            hFunc(TAssignmentEvents::TChangeServiceSettings, OnChangeServiceSettings);
            sFunc(TAssignmentEvents::TScheduleRebalancing, OnScheduleRebalancing);

            // internal
            hFunc(TGossipEvents::TClusterStateResponse, OnClusterStateResponse);
            hFunc(TPrivateEvents::TClusterProbing, OnProbing);
            hFunc(TPrivateEvents::TOneNodeProbingResult, OnOneNodeProbingResult);
            hFunc(TPrivateEvents::TClusterProbingCompleted, OnClusterProbingCompleted);
            hFunc(TPrivateEvents::TAssignmentsLoaded, OnAssignmentsLoaded);
            cFunc(TPrivateEvents::LoadAssignments, LoadAssignments);
            hFunc(TPrivateEvents::TSettingsLoaded, OnSettingsLoaded);
            sFunc(TPrivateEvents::TLoadSettings, LoadSettings);
            sFunc(TPrivateEvents::TReassign, Reassign);
            hFunc(TBalancerEvents::TBalanceResult, OnBalanceResult);

            // also including this, because there could be already scheduled events from the time we were a follower
            hFunc(TPrivateEvents::TLeaderRespondedOnGetAllAssignments, OnLeaderResponseOnGetAllAssignments);
            hFunc(TPrivateEvents::TLeaderRespondedOnGetSlicesByHost, OnLeaderResponseOnGetSlicesByHost);

            sFunc(TPrivateEvents::TMetricsUpdate, OnMetricsUpdate);
        }
    }

    void OnChangeServiceSettings(const TAssignmentEvents::TChangeServiceSettings::TPtr& evPtr) {
        if (!IsLeader_ || !ServiceSettings_) {
            return;
        }

        auto& settings = evPtr->Get()->Settings;
        settings.Service = Service_;
        settings.Cluster = Cluster_;
        settings.Dc = Dc_;

        AM_INFO("saving new settings: " << evPtr->Get()->Settings);
        WriteSettingsToDb(std::move(settings));
    }

    void OnScheduleRebalancing() {
        if (!IsLeader_ || !ServiceSettings_) {
            return;
        }

        Schedule(ServiceSettings_->ReassignmentInterval, new TPrivateEvents::TReassign);
    }

    void OnAssignUniformly() {
        AM_DEBUG("Got an AssignUniformly req");

        // TODO(ivanzhukov): forward a req to a leader if we're a follower
        if (AssnLoadState_ != EDbRecordsLoadState::LOADED || IsAssignmentsRequestToDbInflight_) {
            AM_DEBUG("skipping OnAssignUniformly: still loading from a DB");
            return;
        }

        AM_DEBUG("Assigning uniformly");
        LastReassignment_ = TActivationContext::Now();
        BalanceStats_ = {};

        THPTimer timer;
        auto assn = UniformAssign();
        BalanceStats_.Duration = TDuration::FromValue(timer.Passed() * 1'000'000);

        auto hostToSlices = ConstructHostToSlicesFromAssignments(assn);
        WriteAssignmentsToDb(std::move(hostToSlices), std::move(assn));
    }

    void OnRebalanceOnceByCount() {
        AM_DEBUG("Got a RebalanceOnceByCount req");

        // TODO(ivanzhukov): forward a req to a leader if we're a follower
        AM_DEBUG("Rebalancing by count");
        DoReassign(EReassignmentType::ByCount);
    }

    void OnRebalanceOnceByCpu() {
        AM_DEBUG("Got a RebalanceOnceByCpu req");

        // TODO(ivanzhukov): forward a req to a leader if we're a follower
        DoReassign(EReassignmentType::ByCpu);
    }

    void OnRebalanceOnceByMemory() {
        AM_DEBUG("Got a RebalanceOnceByMemory req");

        // TODO(ivanzhukov): forward a req to a leader if we're a follower
        DoReassign(EReassignmentType::ByMemory);
    }

    void OnValidateShardMoveParams(TAssignmentEvents::TValidateShardMoveParams::TPtr& ev) {
        bool hostValid = (Received_.HostsInfo.find(ev->Get()->HostName) != Received_.HostsInfo.end());
        bool shardIdValid = (Received_.ShardsInfo.find(ev->Get()->ShardId) != Received_.ShardsInfo.end());
        Send(ev->Sender, new TAssignmentEvents::TValidateShardMoveParamsResult{shardIdValid, hostValid});
    }

    void OnGetShardsHostInfo(TAssignmentEvents::TGetShardsHostInfo::TPtr& ev) {
        TString hostName = "not found";
        if (ev->Get()->ShardId != 0) {
            for (const auto&[host, shards]: Received_.HostToNumIds) {
                if (Find(shards, ev->Get()->ShardId) != shards.end()) {
                    hostName = host;
                    break;
                }
            }
        }
        Send(ev->Sender, new TAssignmentEvents::TShardsHostInfoResult{std::move(hostName)});
    }

    void OnGetClusterStatus(TAssignmentEvents::TGetClusterStatus::TPtr& ev) {
        if (!IsLeader_ && LeaderAddress_) {
            TString withoutPort{TStringBuf{LeaderAddress_}.Before(':')};

            Send(ev->Sender, new TAssignmentEvents::TRedirectToLeader{std::move(withoutPort)});
            return;
        } // otherwise just show what we've got until a leader is known

        std::vector<THostLoadInfo> hosts;
        hosts.reserve(LastSeen_.size());

        size_t num = 0;
        for (const auto& [host, lastSeen]: LastSeen_) {
            auto hostInfo = THostLoadInfo{};
            hostInfo.Index = ++num;
            hostInfo.Address = host;
            hostInfo.LastSeen = lastSeen;

            if (auto it = Received_.HostsInfo.find(host); it != Received_.HostsInfo.end()) {
                const auto& load = it->second;

                double hostCpuWorkLoad = 0, hostMemWorkLoad = 0, hostNetWorkLoad = 0;
                const auto& shardIds = Received_.HostToNumIds[host];
                for (auto id: shardIds) {
                    const auto& shardInfo = Received_.ShardsInfo[id];
                    hostCpuWorkLoad += shardInfo.CpuTimeNanos;
                    hostMemWorkLoad += shardInfo.MemoryBytes;
                    hostNetWorkLoad += shardInfo.NetworkBytes;
                }

                hostInfo.NumberOfShards = Received_.HostToNumIds[host].size();
                hostInfo.Cpu = hostCpuWorkLoad;
                hostInfo.CpuAll = static_cast<double>(Max(load.CpuTimeNanos, 1UL));
                hostInfo.Mem = hostMemWorkLoad;
                hostInfo.MemAll = static_cast<double>(Max(load.MemoryBytes, 1UL));
                // TODO: get percentages of network load, after implementing max NetworkBytes
                hostInfo.Net = static_cast<double>(hostNetWorkLoad);
            }

            if (auto it = HostToSlices_.find(host); it != HostToSlices_.end()) {
                hostInfo.NumberOfSlices = it->second.size();
            }

            if (auto it = ClusterMembership_.find(host); it != ClusterMembership_.end()) {
                hostInfo.NodeState = it->second.State;
            }

            hosts.emplace_back(std::move(hostInfo));
        }

        Send(ev->Sender, new TAssignmentEvents::TClusterStatusResult{
                std::move(hosts),
                IsLeader_,
                LeaderAddress_,
                LastReassignment_,
                BalanceStats_,
                ServiceSettings_,
        });
    }

    void OnGetHostInfo(TAssignmentEvents::TGetHostInfo::TPtr& ev) {
        TString hostName = ev->Get()->HostName;
        const auto& shardIds = Received_.HostToNumIds[hostName];
        std::vector<TShardLoadInfo> shards;
        shards.reserve(shardIds.size());

        size_t num = 0;
        for (auto id: shardIds) {
            const auto& shardInfo = Received_.ShardsInfo[id];

            shards.emplace_back(TShardLoadInfo{
                ++num,
                id,
                static_cast<double>(shardInfo.CpuTimeNanos),
                static_cast<double>(Max(Received_.HostsInfo[hostName].CpuTimeNanos, 1UL)),
                static_cast<double>(shardInfo.MemoryBytes),
                static_cast<double>(Max(Received_.HostsInfo[hostName].MemoryBytes, 1UL)),
                // TODO: get percentages of network load, after implementing max NetworkBytes
                static_cast<double>(shardInfo.NetworkBytes)
            });
        }

        std::vector<TString> hostNames;
        hostNames.reserve(Received_.HostsInfo.size());

        for (auto& [host, _]: Received_.HostsInfo) {
            hostNames.emplace_back(host);
        }
        Send(ev->Sender, new TAssignmentEvents::THostInfoResult{shards, hostName, hostNames});
    }

    void OnMetricsUpdate() {
        Schedule(MetricsUpdateInterval_, new TPrivateEvents::TMetricsUpdate);

        if (!IsLeader_) {
            return;
        }

        size_t slicesCnt{0};
        size_t shardsCnt{0};

        for (const auto& [_, slices]: HostToSlices_) {
            slicesCnt += slices.size();
        }

        for (const auto& [_, shards]: Received_.HostToNumIds) {
            shardsCnt += shards.size();
        }

        Metrics_.SlicesCount->Set(slicesCnt);
        Metrics_.ShardsCount->Set(shardsCnt);
        Metrics_.BalancingTookMs->Set(BalanceStats_.Duration.MilliSeconds());
        Metrics_.BalancingMoveKeyChurn->Set(BalanceStats_.MoveKeyChurn);
    }

    void OnMoveShardRequest(const TAssignmentEvents::TMoveShardRequest::TPtr& evPtr) {
        auto& ev = *evPtr->Get();

        if (!IsLeader_) {
            // TODO: forward the request to a leader
            Send(evPtr->Sender, new TAssignmentEvents::TMoveShardResponse{
                    NGrpc::TGrpcStatus{ grpc::StatusCode::UNAVAILABLE, "not a leader" }});
            return;
        }

        if (auto it = ClusterMembership_.find(ev.Host); it == ClusterMembership_.end()) {
            Send(evPtr->Sender, new TAssignmentEvents::TMoveShardResponse{
                    NGrpc::TGrpcStatus{ grpc::StatusCode::NOT_FOUND, "unknown target host: "sv + ev.Host }});
            return;
        } else if (it->second.State != ENodeState::Alive && it->second.State != ENodeState::Suspicious) {
            Send(evPtr->Sender, new TAssignmentEvents::TMoveShardResponse{
                    NGrpc::TGrpcStatus{ grpc::StatusCode::FAILED_PRECONDITION, "the target host is not assignable" }});
            return;
        }

        auto sliceIt = FindSliceToHostContainingNumId(ev.NumId);
        Y_VERIFY(sliceIt != Assignments_.end(), "no slices containing a specified numId");

        if (sliceIt->second[0] == ev.Host) {
            Send(evPtr->Sender, new TAssignmentEvents::TMoveShardResponse{
                    NGrpc::TGrpcStatus{grpc::StatusCode::OK, ""}});
            return;
        }

        auto hostToSlices = HostToSlices_;
        MoveShard(sliceIt->second[0], ev.Host, ev.NumId, sliceIt->first, hostToSlices);

        auto assn = ConstructAssignmentsFromHostToSlices(hostToSlices);
        WriteAssignmentsToDb(std::move(hostToSlices), std::move(assn));
    }

    void OnGetAllAssignmentsRequest(TAssignmentEvents::TGetAllAssignments::TPtr& ev) {
        if (!IsLeader_) {
            if (!LeaderClient_) {
                Send(ev->Sender, new TAssignmentEvents::TGetAllAssignmentsForwardedResponse{
                        std::make_unique<TGetAllAssignmentsResponseOrError>(
                                ::NGrpc::TGrpcStatus(grpc::StatusCode::UNAVAILABLE, "leader is not yet known")),
                        LeaderAddress_});
                return;
            }

            ForwardGetAllAssignmentsRequest(ev->Sender);
            return;
        }

        Send(ev->Sender, new TAssignmentEvents::TGetAllAssignmentsResult{  Assignments_ });
    }

    void OnGetSlicesByHostRequest(TAssignmentEvents::TGetSlicesByHost::TPtr& ev) {
        auto host = ev->Get()->Host;

        if (!IsLeader_) {
            if (!LeaderClient_) {
                Send(ev->Sender, new TAssignmentEvents::TGetSlicesByHostForwardedResponse{
                        std::make_unique<TGetSlicesByHostResponseOrError>(
                                ::NGrpc::TGrpcStatus(grpc::StatusCode::UNAVAILABLE, "leader is not yet known")),
                        LeaderAddress_});
                return;
            }

            ForwardGetSlicesByHostRequest(host, ev->Sender);
            return;
        }

        LastSeen_[host] = TActivationContext::Now();

        if (!ClusterMembership_.contains(host)) {
            // registering a new host
            LoadInfoClusterClient_->Add(host);
            Send(ClusterMembershipActor_, new TGossipEvents::TAddNode{host});
            Send(ev->Sender, new TAssignmentEvents::TGetSlicesByHostResult{NApi::TSlices{}});
            return;
        }

        if (auto it = HostToSlices_.find(host); it != HostToSlices_.end()) {
            Send(ev->Sender, new TAssignmentEvents::TGetSlicesByHostResult{ it->second });
        } else {
            Send(ev->Sender, new TAssignmentEvents::TGetSlicesByHostResult{ TSlices{} });
        }
    }

    void OnProbing(TPrivateEvents::TClusterProbing::TPtr&) {
        Schedule(ProbingOptions_.Interval, new TPrivateEvents::TClusterProbing);

        if (ClusterMembership_.empty() || IsProbingInProgress_) {
            return;
        }

        IsProbingInProgress_ = true;

        TVector<IRequestCtxPtr> requests;

        for (const auto& [address, nodeInfo]: ClusterMembership_) {
            if (nodeInfo.State != ENodeState::Dead) {
                requests.emplace_back(CreateLoadInfoRequestCtx(LoadInfoClusterClient_, address));
            }
        }

        // TODO: move to a constant
        auto reqOptions = TRequestsExecutorOptions()
                .SetMaxRetries(ProbingOptions_.MaxRetries)
                .SetBackoff(ProbingOptions_.Backoff)
                .SetTimeout(ProbingOptions_.Timeout)
                ;

        AM_DEBUG("gathering load info from the cluster. #requests: " << requests.size());

        Register(CreateRequestsExecutorActor(reqOptions, std::move(requests), SelfId()).Release());
    }

    void OnOneNodeProbingResult(TPrivateEvents::TOneNodeProbingResult::TPtr& evPtr) {
        auto* ev = evPtr->Get();

        ProcessLoadInfoResponse(
                ev->HostAddress,
                ev->ResultPtr,
                std::move(ev->ReqCtxPtr));
    }

    void OnClusterProbingCompleted(TPrivateEvents::TClusterProbingCompleted::TPtr&) {
        AM_DEBUG("completed a cluster probing");

        IsProbingInProgress_ = false;
    }

    void OnClusterStateResponse(TGossipEvents::TClusterStateResponse::TPtr& ev) {
        AM_DEBUG("got a cluster state response. cluster size: " << ev->Get()->ClusterState->size());

        ClusterMembership_ = *(ev->Get()->ClusterState);
        UpdateClusterMetrics();
        ClearMissingHosts();

        if (AssnLoadState_ == NONE) {
            LoadAssignments();
        }
    }

    void OnLeaderResponseOnGetAllAssignments(TPrivateEvents::TLeaderRespondedOnGetAllAssignments::TPtr& evPtr) {
        AM_DEBUG("got a GetAllAssignments() response from the leader");

        auto* ev = evPtr->Get();
        Send(
                ev->ReplyTo,
                new TAssignmentEvents::TGetAllAssignmentsForwardedResponse{
                    std::move(ev->Response),
                    std::move(ev->LeaderAddress)});
    }

    void OnLeaderResponseOnGetSlicesByHost(TPrivateEvents::TLeaderRespondedOnGetSlicesByHost::TPtr& evPtr) {
        AM_DEBUG("got a GetSlicesByHost() response from the leader");

        auto* ev = evPtr->Get();
        Send(
                ev->ReplyTo,
                new TAssignmentEvents::TGetSlicesByHostForwardedResponse{
                    std::move(ev->Response),
                    std::move(ev->LeaderAddress)});
    }

// Internal functions
private:
    void ClearMissingHosts() {
        TVector<TNodeEndpoint> missingHosts;

        for (const auto& [host, _]: Received_.HostToNumIds) {
            if (!ClusterMembership_.contains(host)) {
                missingHosts.emplace_back(host);
            }
        }

        for (const auto& host: missingHosts) {
            Received_.HostToNumIds.erase(host);
        }

        missingHosts.clear();

        for (const auto& [host, _]: Received_.HostToSlices) {
            if (!ClusterMembership_.contains(host)) {
                missingHosts.emplace_back(host);
            }
        }

        for (const auto& host: missingHosts) {
            Received_.HostToSlices.erase(host);
        }
    }

    TAssignments::iterator FindSliceToHostContainingNumId(TNumId numId) {
        // find a slice whose End is _greater than or equal to_ numId.
        // see TSliceTransparentComparator for details
        auto it = Assignments_.upper_bound(numId);
        if (it == Assignments_.end() || numId < it->first.Start || numId > it->first.End) {
            return Assignments_.end();
        }

        return it;
    }

    // TODO: rename to "MoveKey", because Slicer will be a general balancer
    // TODO: support HTTP API
    static void MoveShard(
            const TNodeEndpoint& moveFrom,
            const TNodeEndpoint& moveTo,
            TNumId numId,
            TSlice slice,
            TStringMap<TSlices>& hostToSlices)
    {
        auto& fromSlices = hostToSlices[moveFrom];
        auto& toSlices = hostToSlices[moveTo];

        fromSlices.erase(slice);

        if (slice.Start == numId) {
            fromSlices.emplace(slice.Start + 1, slice.End);
        } else if (slice.End == numId) {
            fromSlices.emplace(slice.Start, slice.End - 1);
        } else if (slice.Start < numId && numId < slice.End) {
            fromSlices.emplace(slice.Start, numId - 1);
            fromSlices.emplace(numId + 1, slice.End);
        } else { // start == numId == end
            // already handled
        }

        toSlices.emplace(numId, numId);
    }

    void UpdateClusterMetrics() {
        if (!IsLeader_) {
            return;
        }

        size_t aliveNodes{0};
        size_t suspiciousNodes{0};
        size_t deadNodes{0};
        size_t unknownNodes{0};

        for (const auto& [endpoint, nodeInfo]: ClusterMembership_) {
            switch (nodeInfo.State) {
                case ENodeState::Alive:
                    ++aliveNodes;
                    break;
                case ENodeState::Suspicious:
                    ++suspiciousNodes;
                    break;
                case ENodeState::Dead:
                    ++deadNodes;
                    break;
                case ENodeState::Unknown:
                    ++unknownNodes;
                    break;
            }
        }

        Metrics_.ServiceClusterSize->Set(ClusterMembership_.size());
        Metrics_.ServiceAliveNodes->Set(aliveNodes);
        Metrics_.ServiceSuspiciousNodes->Set(suspiciousNodes);
        Metrics_.ServiceDeadNodes->Set(deadNodes);
        Metrics_.ServiceUnknownNodes->Set(unknownNodes);
    }

    void LoadAssignments() {
        if (AssnLoadState_ != NONE) {
            return;
        }

        if (IsAssignmentsRequestToDbInflight_) {
            AM_DEBUG("skipping reading assignments from a DB because a previous operation is still inflight");
            return;
        }

        AM_INFO("loading assignments from a DB");

        AssnLoadState_ = LOADING;

        AsgnDao_->Load(Service_, Cluster_, Dc_).Subscribe([this, actorSystemPtr = TActorContext::ActorSystem(), actor = SelfId()](auto f) {
            try {
                auto result = f.ExtractValueSync();
                TStringMap<TSlices> converted;
                std::optional<ui32> version;

                for (const auto& item: result) {
                    if (!version.has_value()) {
                        version = item.Version;
                    } else {
                        Y_VERIFY(item.Version == *version, "multiple versions in assignments");
                    }

                    Y_ENSURE(item.Dc == Dc_, "invalid assignment: expected dc: " << Dc_ << ", actual dc: " << item.Dc);
                    Y_ENSURE(item.Cluster == Cluster_, "invalid assignment: expected cluster: " << Cluster_ << ", actual cluster: " << item.Cluster);
                    Y_ENSURE(item.Service == Service_, "invalid assignment: expected service: " << Service_ << ", actual service: " << item.Service);

                    auto& slices = converted[item.Host];

                    // inserting in the reverse order so that insertion is O(1)
                    for (int i = item.HostSlices.slices_size() - 1; i >= 1; i -= 2) {
                        slices.emplace(item.HostSlices.slices(i - 1), item.HostSlices.slices(i));
                    }
                }

                auto assignments = ConstructAssignmentsFromHostToSlices(converted);
                actorSystemPtr->Send(
                        actor,
                        new TPrivateEvents::TAssignmentsLoaded(
                                std::move(converted),
                                std::move(assignments),
                                version.value_or(0)));
            } catch (const yexception& e) {
                actorSystemPtr->Send(actor, new TPrivateEvents::TAssignmentsLoaded(e.what()));
            }
        });
    }

    void LoadSettings() {
        if (SettingsLoadState_ != NONE) {
            return;
        }

        SettingsLoadState_ = LOADING;

        auto as = TActorContext::ActorSystem();
        auto self = SelfId();

        ServiceDao_->Load(NDb::TServiceConfigKey{Service_, Cluster_, Dc_}).Subscribe([as, self](auto f) {
            try {
                auto serviceConfig = f.ExtractValueSync();

                as->Send(self, new TPrivateEvents::TSettingsLoaded(std::move(serviceConfig)));
            } catch (const yexception& e) {
                as->Send(self, new TPrivateEvents::TSettingsLoaded(e.what()));
            }
        });
    }

    TStringMap<TSlices> ComputeHostToSlicesUpdate(const TStringMap<TSlices>& oldAssn, const TStringMap<TSlices>& newAssn) {
        TStringMap<TSlices> result;

        for (const auto& [host, slices]: oldAssn) {
            if (!newAssn.contains(host)) {
                result[host] = {};
            }
        }

        for (const auto& [host, slices]: newAssn) {
            auto it = oldAssn.find(host);

            if (it == oldAssn.end() || it->second != slices) {
                result[host] = slices;
            }
        }

        return result;
    }

    // TODO(ivanzhukov): take diffs and apply (i.e. send via events) only parts that were trully changed
    void WriteAssignmentsToDb(TStringMap<TSlices>&& newHostToSlices, TAssignments&& newAssn) {
        CheckAssignmentsForValidity(newAssn);

        if (IsAssignmentsRequestToDbInflight_) {
            AM_DEBUG("skipping writing assignments to a DB because a previous operation is still inflight");
            return;
        }

        AM_DEBUG("saving assignments to a DB");

        auto update = ComputeHostToSlicesUpdate(HostToSlices_, newHostToSlices);
        if (update.empty()) {
            return;
        }

        AM_TRACE("saving assignments to a DB: " << newAssn);

        TVector<NDb::TAssignment> updateRecords;

        for (const auto& [host, slices]: update) {
            // XXX(ivanzhukov): maybe it'd better to use a separate epoch number incrementing on every rebalancing?
            updateRecords.emplace_back(Service_, Cluster_, Dc_, host, slices, AssignmentsVersion_);
        }

        auto callback = [
                hostToSlices{std::move(newHostToSlices)},
                assn{std::move(newAssn)},
                as{TActorContext::ActorSystem()},
                self{SelfId()},
                version{AssignmentsVersion_}
        ](const NDb::TAsyncVoid& f) mutable {
            try {
                f.GetValueSync();
                as->Send(self, new TPrivateEvents::TAssignmentsLoaded(std::move(hostToSlices), std::move(assn), version));
            } catch (const yexception& e) {
                // FIXME(ivanzhukov): distinguish between a legitimate write error and a version mismatch
                as->Send(self, new TPrivateEvents::TAssignmentsLoaded(e.what()));
            }
        };

        AsgnDao_->Update(updateRecords).Subscribe(callback);
        IsAssignmentsRequestToDbInflight_ = true;
    }

    void WriteSettingsToDb(NDb::TServiceConfig&& settings) {
        if (IsSettingsRequestToDbInflight_) {
            AM_DEBUG("skipping a settings write to a DB because a previous operation is still inflight");
            return;
        }

        AM_DEBUG("saving the settings to the DB: " << settings);

        auto callback = [settings, actorSystemPtr = TActorContext::ActorSystem(), actor = SelfId()](const NDb::TAsyncVoid& f) mutable {
            try {
                f.GetValueSync();
                actorSystemPtr->Send(actor, new TPrivateEvents::TSettingsLoaded(std::move(settings)));
            } catch (const yexception& e) {
                actorSystemPtr->Send(actor, new TPrivateEvents::TSettingsLoaded(e.what()));
            }
        };

        ServiceDao_->Update(settings).Subscribe(callback);
        IsSettingsRequestToDbInflight_ = true;
    }

    void OnAssignmentsLoaded(TPrivateEvents::TAssignmentsLoaded::TPtr& ev) {
        IsAssignmentsRequestToDbInflight_ = false;

        if (!IsLeader_) {
            return;
        }

        if (ev->Get()->Result.Fail()) {
            AssnLoadState_ = NONE;

            Schedule(TDuration::Seconds(1), new TPrivateEvents::TLoadAssignments);
            AM_ERROR("failed to load assignments: " << ev->Get()->Result.Error());
            return;
        }

        auto& res = ev->Get()->Result.Value();

        if (res.empty()) {
            AM_INFO("got empty assignments from a DB. reassigning uniformly");

            auto assn = UniformAssign();

            if (assn.empty()) {
                AssnLoadState_ = NONE;
                Schedule(TDuration::Seconds(1), new TPrivateEvents::TLoadAssignments);
                return;
            }

            auto hostToSlices = ConstructHostToSlicesFromAssignments(assn);
            WriteAssignmentsToDb(std::move(hostToSlices), std::move(assn));
            return;
        }

        AM_INFO("loaded assignments from a DB");
        AssnLoadState_ = LOADED;

        UpdateAssignments(res, ev->Get()->Assignments, ev->Get()->Version);
    }

    void OnSettingsLoaded(TPrivateEvents::TSettingsLoaded::TPtr& ev) {
        IsSettingsRequestToDbInflight_ = false;

        if (!IsLeader_) {
            return;
        }

        if (ev->Get()->Result.Fail()) {
            SettingsLoadState_ = NONE;

            Schedule(TDuration::Seconds(1), new TPrivateEvents::TLoadSettings);
            AM_ERROR("failed to load settings: " << ev->Get()->Result.Error());
            return;
        }

        auto& res = ev->Get()->Result.Value();

        if (!res) {
            WriteSettingsToDb(NDb::TServiceConfig(Service_, Cluster_, Dc_));
            return;
        }

        if (!ServiceSettings_ && !res->IsFrozen) { // we are a leader and we load a config for the first time
            Schedule(REASSIGNMENT_FUSE + res->ReassignmentInterval, new TPrivateEvents::TReassign);
        }

        SettingsLoadState_ = LOADED;
        ServiceSettings_ = std::move(*res);
    }

    void UpdateAssignments(TStringMap<TSlices>& hostToSlices, TAssignments& assn, ui32 version) {
        AssignmentsVersion_ = Max(AssignmentsVersion_ + 1, version + 1);

        AM_INFO("applying assignments");
        AM_TRACE("applying assignments: " << assn);

        CheckAssignmentsForValidity(assn);

        HostToSlices_ = std::move(hostToSlices);
        Assignments_ = std::move(assn);
    }

    TAssignments UniformAssign() {
        size_t numOfAliveNodes = 0;

        for (const auto& [_, node]: ClusterMembership_) {
            switch (node.State) {
                case ENodeState::Dead:
                case ENodeState::Unknown:
                    continue;
                case ENodeState::Suspicious:
                case ENodeState::Alive:
                    ++numOfAliveNodes;
                    break;
            }
        }

        if (numOfAliveNodes == 0) {
            return {};
        }

        ui64 numOfKeys = std::numeric_limits<TNumId>::max(); // 2**32 - 1, because 0 is not a correct NumId
        ui64 sliceSize = std::floor(numOfKeys / static_cast<double>(numOfAliveNodes));
        static_assert(
                std::numeric_limits<TNumId>::max() <= std::numeric_limits<ui32>::max(),
                "cannot use a modulo here due to a possible overflow. Rewrite to a precise calculation");
        ui64 lastSliceSize = sliceSize + (numOfKeys % numOfAliveNodes);

        TNumId sliceStart = 1; // NumId starts from 1
        TNumId sliceEnd;
        size_t i = 0;

        TAssignments assign;

        for (const auto& [nodeEndpoint, nodeInfo]: ClusterMembership_) {
            switch (nodeInfo.State) {
                case ENodeState::Dead:
                case ENodeState::Unknown:
                    continue;
                case ENodeState::Suspicious:
                case ENodeState::Alive:
                    break; // from the switch and execute the code below
            }

            sliceEnd = (i == numOfAliveNodes - 1)
                    ? (sliceStart + lastSliceSize - 1)
                    : (sliceStart + sliceSize - 1);
            ++i;

            TSlice slice{sliceStart, sliceEnd};
            THosts hosts(1, nodeEndpoint);

            assign.emplace(slice, hosts);

            sliceStart = sliceEnd + 1;
        }

        return assign;
    }

    void Reassign() {
        if (ServiceSettings_->IsFrozen) {
            AM_DEBUG("skipping reassignment because it is disabled");
            return;
        }

        DoReassign(ServiceSettings_->ReassignmentType);

        Y_DEFER {
            Schedule(ServiceSettings_->ReassignmentInterval, new TPrivateEvents::TReassign);
        };
    }

    void DoReassign(EReassignmentType rebalancingType) {
        LastReassignment_ = TActivationContext::Now();

        if (AssnLoadState_ != EDbRecordsLoadState::LOADED || IsAssignmentsRequestToDbInflight_) {
            AM_DEBUG("skipping reassignment: still loading from a DB");
            return;
        }
        // TODO(ivanzhukov): check for an already going rebalancing

        Register(CreateBalancer(
                Service_,
                SelfId(),
                rebalancingType,
                ClusterMembership_,
                HostToSlices_,
                Received_,
                *ServiceSettings_).release());
    }

    void OnBalanceResult(const TBalancerEvents::TBalanceResult::TPtr& evPtr) {
        auto& ev = *evPtr->Get();

        BalanceStats_ = std::move(ev.Stats);
        WriteAssignmentsToDb(std::move(ev.HostToSlices), std::move(ev.Assignments));
    }

    void ProcessLoadInfoResponse(
            const TNodeEndpoint& hostAddress,
            const TPrivateEvents::TOneNodeProbingResult::TResultPtr& resultPtr,
            IRequestCtxPtr reqCtxPtr)
    {
        auto& valueOrError = *resultPtr;

        if (valueOrError.Fail()) {
            AM_ERROR("failed request " << reqCtxPtr->What()
                             << " with an error: " << valueOrError.Error().MessageString());
            return;
        }

        auto& value = valueOrError.Value();

        Received_.HostToNumIds[hostAddress] = {};
        auto& hostNumIds = Received_.HostToNumIds[hostAddress];

        // Note: ingestor sends already EWMAed metrics
        const auto& shards = value.shards_load();
        for (int i = 0; i != shards.shard_ids_size(); ++i) {
            auto numId = shards.shard_ids(i);

            Received_.ShardsInfo[numId] = {
                    .CpuTimeNanos = shards.cpu_time_nanos(i),
                    .MemoryBytes = shards.memory_bytes(i),
                    .NetworkBytes = shards.network_bytes(i),
            };
            hostNumIds.emplace(numId);
        }

        const auto& host = value.host_load();
        Received_.HostsInfo[hostAddress] = {
                .CpuTimeNanos = host.cpu_time_nanos(),
                .MemoryBytes = host.memory_bytes(),
                .NetworkBytes = host.network_bytes(),
        };

        Received_.HostToSlices[hostAddress] = {};
        auto& slices = Received_.HostToSlices[hostAddress];

        for (int i = 0; i != value.slices_starts_size(); ++i) {
            slices.emplace(value.slices_starts(i), value.slices_ends(i));
        }
    }

    void ForwardGetAllAssignmentsRequest(const TActorId& replyTo) {
        // TODO(ivanzhukov): add a cache
        TVector<IRequestCtxPtr> requests{
            CreateForwardedGetAllAssignmentsRequestCtx(Service_, LeaderClient_, LeaderAddress_, replyTo)};

        AM_DEBUG("starting a " << requests[0]->What());

        auto reqOptions = TRequestsExecutorOptions()
                // TODO(ivanzhukov): take values from a config
                .SetMaxRetries(1)
                .SetBackoff(TDuration::Zero())
                .SetTimeout(TDuration::Zero())
                ;

        Register(CreateRequestsExecutorActor(reqOptions, std::move(requests), SelfId()).Release());
    }

    void ForwardGetSlicesByHostRequest(TString host, const TActorId& replyTo) {
        // TODO(ivanzhukov): add a cache
        TVector<IRequestCtxPtr> requests{
            CreateForwardedGetSlicesByHostRequestCtx(Service_, std::move(host), LeaderClient_, LeaderAddress_, replyTo)
        };

        AM_DEBUG("starting a " << requests[0]->What());

        auto reqOptions = TRequestsExecutorOptions()
                // TODO(ivanzhukov): take values from a config
                .SetMaxRetries(1)
                .SetBackoff(TDuration::Zero())
                .SetTimeout(TDuration::Zero())
        ;

        Register(CreateRequestsExecutorActor(reqOptions, std::move(requests), SelfId()).Release());
    }

private:
    // leader state listeners
    void OnBecomeLeader() override {
        AM_DEBUG("became the leader");
        IsLeader_ = true;
        Metrics_.IsLeader->Set(1);
        LeaderAddress_ = {};
        LeaderClient_ = {};

        AssnLoadState_ = NONE;
        LoadAssignments();

        ServiceSettings_ = {};
        SettingsLoadState_ = NONE;
        LoadSettings();

        Become(&TThis::LeaderState);
    }

    void OnBecomeFollower() override {
        AM_DEBUG("became a follower");
        IsLeader_ = false;
        Metrics_.Reset();

        Become(&TThis::FollowerState);
    }

    void OnLeaderChanged(TString leaderId, ui64 lockEpoch) override {
        if (static_cast<int>(lockEpoch) <= LockEpoch_) {
            AM_DEBUG("got a stale lock acquiring. leaderId: " << leaderId << "; lockEpoch: " << lockEpoch
                    << "; current epoch: " << LockEpoch_);
            return;
        }

        // TODO(ivanzhukov): send a metric
        AM_DEBUG("leader has changed. leaderId: " << leaderId << "; lockEpoch: " << lockEpoch);

        LeaderAddress_ = leaderId;
        LeaderAddress_ += ':';
        LeaderAddress_ += GrpcPort_;
        LockEpoch_ = lockEpoch;

        if (LeaderClient_ = SlicerClients_->Get(LeaderAddress_); !LeaderClient_) {
            // TODO(ivanzhukov): new node in a cluster -> create a new client
        }
    }

    void OnError(const TString& message) override {
        // TODO(ivanzhukov): send a metric
        AM_WARN("got an error while trying to acquire a lock: " << message);
    }

private:
    TString Service_;
    SlicerConfig Config_;
    struct {
        TDuration Interval;
        size_t MaxRetries{0};
        TDuration Backoff;
        TDuration Timeout;
    } ProbingOptions_;
    TActorId ClusterMembershipActor_;

    IDistributedLockPtr Lock_;
    TDuration LeaderElectionInterval_;
    ISlicerClusterClientPtr SlicerClients_;
    ISlicerClient* LeaderClient_{nullptr};
    TString LeaderAddress_;
    bool IsLeader_{false};
    int LockEpoch_{-1};
    TStringBuf GrpcPort_;
    ui32 AssignmentsVersion_{0};

    TClusterMembershipState ClusterMembership_;
    // Data received from services (ingestor, memstore, etc.)
    THostReportedInfo Received_;
    /**
     * A time at which Slicer received a GetSlicesByHost request from a node
     */
    std::unordered_map<TString, TInstant> LastSeen_;

    // TODO: encapsulate in one data structure that changes both:
    //  1. the assignments
    TAssignments Assignments_;
    //  2. and the mapping
    TStringMap<TSlices> HostToSlices_;

    NMonitoring::TMetricRegistry& Registry_;
    TMetrics Metrics_;
    ILoadInfoClusterClientPtr LoadInfoClusterClient_;

    std::optional<NDb::TServiceConfig> ServiceSettings_;
    // TODO(ivanzhukov): either call everything Rebalancing* or Reassignment*, not both
    TInstant LastReassignment_;
    TBalanceStats BalanceStats_{};
    TDuration MetricsUpdateInterval_;
    bool IsProbingInProgress_{false};

    enum EDbRecordsLoadState {
        NONE,
        LOADING,
        LOADED
    };
    EDbRecordsLoadState AssnLoadState_{NONE};
    EDbRecordsLoadState SettingsLoadState_{NONE};

    NDb::IAssignmentDaoPtr AsgnDao_;
    bool IsAssignmentsRequestToDbInflight_{false};
    std::shared_ptr<NDb::IServiceConfigDao> ServiceDao_;
    bool IsSettingsRequestToDbInflight_{false};
    TString Dc_;
    TString Cluster_;
};
#undef AM_TRACE
#undef AM_DEBUG
#undef AM_INFO
#undef AM_WARN
#undef AM_ERROR
#undef AM_CRIT

} // namespace

THolder<IActor> CreateAssignmentManager(
        TString serviceName,
        const SlicerConfig& config,
        TActorId clusterMembershipActor,
        IDistributedLockPtr lock,
        ISlicerClusterClientPtr slicerClients,
        NMonitoring::TMetricRegistry& registry,
        TString clientId,
        NDb::IAssignmentDaoPtr assnDao,
        std::shared_ptr<NDb::IServiceConfigDao> serviceDao)
{
    return MakeHolder<TAssignmentManager>(
            std::move(serviceName),
            config,
            clusterMembershipActor,
            std::move(lock),
            std::move(slicerClients),
            registry,
            std::move(clientId),
            std::move(assnDao),
            std::move(serviceDao));
}

} // namespace NSolomon::NSlicer::NApi
