#pragma once

#include <solomon/services/slicer/lib/balancer/balancer.h>
#include <solomon/services/slicer/lib/common/reassignment_type.h>
#include <solomon/services/slicer/lib/db/service_config.h>
#include <solomon/services/slicer/lib/events/event_slots.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/cluster_membership/node_states.h>
#include <solomon/libs/cpp/clients/slicer/api_types.h>
#include <solomon/libs/cpp/clients/slicer/slicer_client.h>

#include <library/cpp/actors/core/event_local.h>

namespace NSolomon::NSlicer {

struct TSlicesResult {
    NApi::TSlices Slices;
};

struct THostLoadInfo {
    size_t Index;
    TString Address;
    size_t NumberOfShards{0};
    size_t NumberOfSlices{0};
    double Cpu{0};
    double CpuAll{0};
    double Mem{0};
    double MemAll{0};
    double Net{0};
    NClusterMembership::ENodeState NodeState{NClusterMembership::ENodeState::Unknown};
    TInstant LastSeen;
};

struct TShardLoadInfo {
    size_t Index;
    NApi::TNumId ShardId;
    double Cpu;
    double CpuUsedOnHost;
    double Mem;
    double MemUsedOnHost;
    double Net;
};

enum class EOrdered {
    Ascending,
    Descending,
};

class TAssignmentEvents: private TEventSlot<EEventSpace::Slicer, ES_ASSIGNMENTS> {
    enum {
        GetAllAssignments = SpaceBegin,
        GetSlicesByHost,
        GetAllAssignmentsResult,
        GetSlicesByHostResult,
        GetAllAssignmentsForwardedResponse,
        GetSlicesByHostForwardedResponse,
        MoveShardRequest,
        MoveShardResponse,
        EvTimeout,
        GetClusterStatus,
        GetShardsHostInfo,
        ClusterStatusResult,
        HostInfoResult,
        ShardsHostInfoResult,
        GetHostInfo,
        ValidateShardMoveParams,
        ValidateShardMoveParamsResult,
        AssignUniformly,
        RebalanceOnceByCount,
        RebalanceOnceByCpu,
        RebalanceOnceByMemory,
        ScheduleRebalancing,
        ChangeServiceSettings,
        RedirectToLeader,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    struct TGetAllAssignments: public ::NActors::TEventLocal<TGetAllAssignments, GetAllAssignments> {
        TString Service;
    };

    struct TGetSlicesByHost: public ::NActors::TEventLocal<TGetSlicesByHost, GetSlicesByHost> {
        TString Service;
        TString Host;

        TGetSlicesByHost(TString service, TString host)
            : Service{std::move(service)}
            , Host{std::move(host)}
        {}
    };

    struct TGetAllAssignmentsResult: public ::NActors::TEventLocal<TGetAllAssignmentsResult, GetAllAssignmentsResult> {
        std::map<NApi::TSlice, NApi::THosts, NApi::TSliceTransparentComparator> Assignments;

        TGetAllAssignmentsResult(decltype(Assignments) assignments)
            : Assignments{std::move(assignments)}
        {}
    };

    struct TGetSlicesByHostResult: public ::NActors::TEventLocal<TGetSlicesByHostResult, GetSlicesByHostResult> {
        TErrorOr<TSlicesResult, ::NGrpc::TGrpcStatus> SlicesResult;

        explicit TGetSlicesByHostResult(::NGrpc::TGrpcStatus&& err)
            : SlicesResult{err}
        {}

        explicit TGetSlicesByHostResult(NApi::TSlices slices)
            : SlicesResult{TSlicesResult{
                  .Slices = std::move(slices),
              }}
        {}
    };

    struct TGetAllAssignmentsForwardedResponse:
        public ::NActors::TEventLocal<TGetAllAssignmentsForwardedResponse, GetAllAssignmentsForwardedResponse>
    {
        NSolomon::NSlicerClient::TGetAllAssignmentsResponseOrErrorPtr Response;
        TString LeaderAddress;

        TGetAllAssignmentsForwardedResponse(decltype(Response) response, TString leaderAddress)
            : Response{std::move(response)}
            , LeaderAddress{std::move(leaderAddress)}
        {}
    };

    struct TGetSlicesByHostForwardedResponse:
        public ::NActors::TEventLocal<TGetSlicesByHostForwardedResponse, GetSlicesByHostForwardedResponse>
    {
        NSolomon::NSlicerClient::TGetSlicesByHostResponseOrErrorPtr Response;
        TString LeaderAddress;

        TGetSlicesByHostForwardedResponse(decltype(Response) response, TString leaderAddress)
            : Response{std::move(response)}
            , LeaderAddress{std::move(leaderAddress)}
        {}
    };

    struct TMoveShardRequest: public ::NActors::TEventLocal<TMoveShardRequest, MoveShardRequest> {
        TString Service;
        NApi::TNumId NumId;
        TString Host;

        TMoveShardRequest(TString service, NApi::TNumId numId, TString host)
            : Service{std::move(service)}
            , NumId{numId}
            , Host{std::move(host)}
        {}
    };

    struct TMoveShardResponse: public ::NActors::TEventLocal<TMoveShardResponse, MoveShardResponse> {
        ::NGrpc::TGrpcStatus Status;

        explicit TMoveShardResponse(::NGrpc::TGrpcStatus status)
            : Status{std::move(status)}
        {}
    };

    struct TGetClusterStatus: public ::NActors::TEventLocal<TGetClusterStatus, GetClusterStatus> {
        TString Service;

        explicit TGetClusterStatus(TString service) noexcept
            : Service{std::move(service)}
        {}
    };

    struct TGetShardsHostInfo: public ::NActors::TEventLocal<TGetShardsHostInfo, GetShardsHostInfo> {
        TGetShardsHostInfo(TString service, NApi::TNumId shardId)
            : Service{std::move(service)}
            , ShardId{shardId}
        {
        }

        TString Service;
        NApi::TNumId ShardId;
    };

    struct TClusterStatusResult: public ::NActors::TEventLocal<TClusterStatusResult, ClusterStatusResult> {
        TClusterStatusResult(
                std::vector<THostLoadInfo> hosts,
                bool isLeader,
                TString leaderAddress,
                TInstant lastReassignment,
                TBalanceStats balanceStats,
                std::optional<NDb::TServiceConfig> serviceSettings)
            : Hosts{std::move(hosts)}
            , IsLeader{isLeader}
            , LeaderAddress{std::move(leaderAddress)}
            , LastReassignment{lastReassignment}
            , BalanceStats{balanceStats}
            , ServiceSettings{std::move(serviceSettings)}
        {
        }

        std::vector<THostLoadInfo> Hosts;
        bool IsLeader;
        TString LeaderAddress;
        TInstant LastReassignment;
        TBalanceStats BalanceStats;
        std::optional<NDb::TServiceConfig> ServiceSettings;
    };

    struct TGetHostInfo: public ::NActors::TEventLocal<TGetHostInfo, GetHostInfo> {
        TGetHostInfo(TString service, TString hostName)
            : Service{std::move(service)}
            , HostName{std::move(hostName)}
        {
        }

        TString Service;
        TString HostName;
    };

    struct TValidateShardMoveParams: public ::NActors::TEventLocal<TValidateShardMoveParams, ValidateShardMoveParams> {
        TValidateShardMoveParams(TString service, NApi::TNumId shardId, TString hostName) noexcept
            : Service{std::move(service)}
            , ShardId{shardId}
            , HostName{std::move(hostName)}
        {
        }

        TString Service;
        NApi::TNumId ShardId;
        TString HostName;
    };

    struct TValidateShardMoveParamsResult:
        public ::NActors::TEventLocal<TValidateShardMoveParamsResult, ValidateShardMoveParamsResult> {
        TValidateShardMoveParamsResult(bool shardIdValid, bool hostValid) noexcept
            : ShardIdValid{shardIdValid}
            , HostValid{hostValid}
        {
        }

        bool ShardIdValid;
        bool HostValid;
    };

    struct THostInfoResult: public ::NActors::TEventLocal<THostInfoResult, HostInfoResult> {
        THostInfoResult(std::vector<TShardLoadInfo> shards, TString hostName, std::vector<TString> hostNames)
            : HostName{std::move(hostName)}
            , Shards{std::move(shards)}
            , HostNames{std::move(hostNames)}
        {
        }

        TString HostName;
        std::vector<TShardLoadInfo> Shards;
        std::vector<TString> HostNames;
    };

    struct TShardsHostInfoResult: public ::NActors::TEventLocal<TShardsHostInfoResult, ShardsHostInfoResult> {
        explicit TShardsHostInfoResult(TString hostName)
            : HostName{std::move(hostName)}
        {
        }

        TString HostName;
    };

    struct TAssignUniformly: public NActors::TEventLocal<TAssignUniformly, AssignUniformly> {
        TString Service;

        explicit TAssignUniformly(TString service) noexcept
            : Service{std::move(service)}
        {
        }
    };

    struct TRebalanceOnceByCount: public ::NActors::TEventLocal<TRebalanceOnceByCount, RebalanceOnceByCount> {
        TString Service;

        explicit TRebalanceOnceByCount(TString service) noexcept
            : Service{std::move(service)}
        {
        }
    };

    struct TRebalanceOnceByCpu: public ::NActors::TEventLocal<TRebalanceOnceByCpu, RebalanceOnceByCpu> {
        TString Service;

        explicit TRebalanceOnceByCpu(TString service) noexcept
            : Service{std::move(service)}
        {
        }
    };

    struct TRebalanceOnceByMemory: public ::NActors::TEventLocal<TRebalanceOnceByMemory, RebalanceOnceByMemory> {
        TString Service;

        explicit TRebalanceOnceByMemory(TString service) noexcept
                : Service{std::move(service)}
        {
        }
    };

    struct TScheduleRebalancing: public NActors::TEventLocal<TScheduleRebalancing, ScheduleRebalancing> {
        TString Service;

        explicit TScheduleRebalancing(TString service)
            : Service{std::move(service)}
        {
        }
    };

    struct TChangeServiceSettings: public NActors::TEventLocal<TChangeServiceSettings, ChangeServiceSettings> {
        TString Service;
        NDb::TServiceConfig Settings;

        TChangeServiceSettings(TString service, NDb::TServiceConfig settings)
            : Service{std::move(service)}
            , Settings{std::move(settings)}
        {
        }
    };

    struct TRedirectToLeader: public NActors::TEventLocal<TRedirectToLeader, RedirectToLeader> {
        TString LeaderFqdn;

        explicit TRedirectToLeader(TString leaderFqdn) noexcept
            : LeaderFqdn{std::move(leaderFqdn)}
        {
        }
    };
};

class TServiceManagerEvents: private TEventSlot<EEventSpace::Slicer, ES_SERVICE_MANAGER> {
    enum {
        GetServices = SpaceBegin,
        GetServicesResponse,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    struct TGetServices: public ::NActors::TEventLocal<TGetServices, GetServices> {
    };

    struct TGetServicesResponse: public ::NActors::TEventLocal<TGetServicesResponse, GetServicesResponse> {
        TVector<TString> Services;

        explicit TGetServicesResponse(TVector<TString> services)
            : Services{std::move(services)}
        {
        }
    };
};

} // namespace NSolomon::NSlicer
