#pragma once

#include <solomon/services/slicer/lib/common/assignments.h>
#include <solomon/services/slicer/lib/db/assignment_dao.h>
#include <solomon/services/slicer/lib/db/service_config.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/util/requests_executor.h>
#include <solomon/libs/cpp/clients/load_info/load_service_client.h>
#include <solomon/libs/cpp/cluster_membership/events.h>
#include <solomon/libs/cpp/clients/slicer/slicer_client.h>

namespace NSolomon::NSlicer::NApi {

struct TPrivateEvents: private NSolomon::TPrivateEvents {
    enum {
        Assign = SpaceBegin,
        ClusterProbing,
        Reassign,

        OneNodeProbingResult,
        ClusterProbingCompleted,

        LeaderRespondedOnGetAllAssignments,
        LeaderRespondedOnGetSlicesByHost,

        MetricsUpdate,

        LoadAssignments,
        AssignmentsLoaded,
        AssignmentsUpdated,
        LoadSettings,
        SettingsLoaded,

        End,
    };

    static_assert(End < SpaceEnd, "too many event types");

    struct TClusterProbing: NActors::TEventLocal<TClusterProbing, ClusterProbing> {
    };

    struct TOneNodeProbingResult: NActors::TEventLocal<TOneNodeProbingResult, OneNodeProbingResult> {
        using TResultPtr = std::unique_ptr<NLoadInfo::TAsyncLoadInfoResponse::value_type>;

        IRequestCtxPtr ReqCtxPtr;
        TResultPtr ResultPtr;
        NClusterMembership::TNodeEndpoint HostAddress;

        TOneNodeProbingResult(
                IRequestCtxPtr reqCtxPtr,
                TResultPtr resultPtr,
                NClusterMembership::TNodeEndpoint hostAddress)
            : ReqCtxPtr{std::move(reqCtxPtr)}
            , ResultPtr{std::move(resultPtr)}
            , HostAddress{std::move(hostAddress)}
        {}
    };

    struct TClusterProbingCompleted: NActors::TEventLocal<TClusterProbingCompleted, ClusterProbingCompleted> {
    };

    struct TLeaderRespondedOnGetAllAssignments:
        NActors::TEventLocal<TLeaderRespondedOnGetAllAssignments, LeaderRespondedOnGetAllAssignments>
    {
        NSlicerClient::TGetAllAssignmentsResponseOrErrorPtr Response;
        TString LeaderAddress;
        NActors::TActorId ReplyTo;

        TLeaderRespondedOnGetAllAssignments(
                NSlicerClient::TGetAllAssignmentsResponseOrErrorPtr response,
                TString leaderAddress,
                NActors::TActorId replyTo)
            : Response{std::move(response)}
            , LeaderAddress{std::move(leaderAddress)}
            , ReplyTo{replyTo}
        {}
    };

    struct TLeaderRespondedOnGetSlicesByHost:
        NActors::TEventLocal<TLeaderRespondedOnGetSlicesByHost, LeaderRespondedOnGetSlicesByHost>
    {
        NSlicerClient::TGetSlicesByHostResponseOrErrorPtr Response;
        TString LeaderAddress;
        NActors::TActorId ReplyTo;

        TLeaderRespondedOnGetSlicesByHost(
                NSlicerClient::TGetSlicesByHostResponseOrErrorPtr response,
                TString leaderAddress,
                NActors::TActorId replyTo)
            : Response{std::move(response)}
            , LeaderAddress{std::move(leaderAddress)}
            , ReplyTo{replyTo}
        {}
    };

    struct TMetricsUpdate: NActors::TEventLocal<TMetricsUpdate, MetricsUpdate> {
    };

    struct TAssignmentsLoaded: public NActors::TEventLocal<TAssignmentsLoaded, AssignmentsLoaded> {
        TAssignmentsLoaded(TStringMap<TSlices> hostToSlices, TAssignments assn, ui32 version)
            : Result(std::move(hostToSlices))
            , Assignments{std::move(assn)}
            , Version{version}
        {
        }

        explicit TAssignmentsLoaded(TString error)
            : Result(std::move(error))
        {
        }

        TErrorOr<TStringMap<TSlices>, TString> Result;
        TAssignments Assignments;
        ui32 Version;
    };

    struct TLoadAssignments: public NActors::TEventLocal<TLoadAssignments, LoadAssignments> {
    };

    struct TLoadSettings: public NActors::TEventLocal<TLoadSettings, LoadSettings> {
    };

    struct TSettingsLoaded: public NActors::TEventLocal<TSettingsLoaded, SettingsLoaded> {
        TSettingsLoaded(std::optional<NDb::TServiceConfig> data)
            : Result(std::move(data))
        {
        }

        TSettingsLoaded(TString error)
            : Result(std::move(error))
        {
        }

        TErrorOr<std::optional<NDb::TServiceConfig>, TString> Result;
    };

    struct TReassign: public NActors::TEventLocal<TReassign, Reassign> {
    };
};

} // namespace NSolomon::NSlicer::NApi
