#include "request_ctx_fwd_get_all_assn.h"

namespace NSolomon::NSlicer::NApi {
namespace {

class TForwardedGetAllAssignmentsRequestCtx: public IRequestCtx {
public:
    TForwardedGetAllAssignmentsRequestCtx(
            const TString& service,
            NSolomon::NSlicerClient::ISlicerClient* slicerClient,
            const TString& leaderAddress,
            NActors::TActorId replyTo)
        : Service_{std::move(service)}
        , SlicerClient_{std::move(slicerClient)}
        , LeaderAddress_{leaderAddress}
        , ReplyTo_{std::move(replyTo)}
    {
        What_ = TStringBuilder() << "[GetAllAssignments(\"" << Service_ << "\") request forwarded to " << leaderAddress << "]";
    }

private:
    NThreading::TFuture<void> PerformRequest() override {
        ++TimesPerformed_;
        return HandleResult(SlicerClient_->GetAllAssignments(Service_));
    }

    NThreading::TFuture<void> HandleResult(const NSolomon::NSlicerClient::TAsyncGetAllAssignmentsResponse& result) {
        return result
            .Subscribe([this, reqCtxPtr{shared_from_this()}](NSolomon::NSlicerClient::TAsyncGetAllAssignmentsResponse future) {
                try {
                    Result_ = future.ExtractValueSync();
                    ShouldRetry_ = Result_->Fail();
                } catch (...) {
                    ShouldRetry_ = true;
                    Result_ = std::make_unique<NSolomon::NSlicerClient::TGetAllAssignmentsResponseOrError>(
                        NGrpc::TGrpcStatus{CurrentExceptionMessage(), grpc::StatusCode::INTERNAL, true});
                }
            })
            .IgnoreResult();
    }

    bool ShouldRetry() const override {
        // TODO(ivanzhukov): take into account cases when it shouldn't be retried (exceptions, network overload, etc.)
        return ShouldRetry_;
    }

    TString What() const override {
        return What_;
    }

    size_t TimesPerformed() const override {
        return TimesPerformed_;
    }

    std::unique_ptr<NActors::IEventBase> MakeReplyEvent() override {
        return {};
    }

    std::unique_ptr<NActors::IEventBase> MakeCompletionEvent() override {
        return std::make_unique<TPrivateEvents::TLeaderRespondedOnGetAllAssignments>(
            std::move(Result_), std::move(LeaderAddress_), std::move(ReplyTo_));
    }

private:
    const TString& Service_;
    NSolomon::NSlicerClient::ISlicerClient* SlicerClient_;
    TString LeaderAddress_;
    NActors::TActorId ReplyTo_;

    NSolomon::NSlicerClient::TGetAllAssignmentsResponseOrErrorPtr Result_;
    TString What_;
    size_t TimesPerformed_{0};
    bool ShouldRetry_{false};
};

} // namespace

std::shared_ptr<IRequestCtx> CreateForwardedGetAllAssignmentsRequestCtx(
    const TString& service,
    NSolomon::NSlicerClient::ISlicerClient* slicerClient,
    TString leaderAddress,  // NOLINT(performance-unnecessary-value-param): false positive
    NActors::TActorId replyTo)
{
    return std::make_shared<TForwardedGetAllAssignmentsRequestCtx>(
        service,
        std::move(slicerClient),
        std::move(leaderAddress), replyTo);
}

} // namespace NSolomon::NSlicer::NApi
