#include "request_ctx_fwd_get_slices_by_host.h"

namespace NSolomon::NSlicer::NApi {
namespace {

class TForwardedGetSlicesByHostRequestCtx: public IRequestCtx {
public:
    TForwardedGetSlicesByHostRequestCtx(
            const TString& service,
            TString host,
            NSolomon::NSlicerClient::ISlicerClient* slicerClient,
            const TString& leaderAddress,
            NActors::TActorId replyTo)
        : Service_{service}
        , Host_{std::move(host)}
        , SlicerClient_{std::move(slicerClient)}
        , LeaderAddress_{leaderAddress}
        , ReplyTo_{std::move(replyTo)}
    {
        What_ = TStringBuilder() << "[GetSlicesByHost(\"" << Service_ << "\", \"" << Host_ << "\")"
                                 << " request forwarded to " << leaderAddress << "]";
    }

private:
    NThreading::TFuture<void> PerformRequest() override {
        ++TimesPerformed_;
        return HandleResult(SlicerClient_->GetSlicesByHost(Service_, Host_));
    }

    NThreading::TFuture<void> HandleResult(const NSlicerClient::TAsyncGetSlicesByHostResponse& result) {
        return result
            .Subscribe([this, reqCtxPtr{shared_from_this()}](NSlicerClient::TAsyncGetSlicesByHostResponse future) {
                try {
                    Result_ = future.ExtractValueSync();
                    ShouldRetry_ = Result_->Fail();
                } catch (...) {
                    ShouldRetry_ = true;
                    Result_ = std::make_unique<NSlicerClient::TGetSlicesByHostResponseOrError>(
                        NGrpc::TGrpcStatus{CurrentExceptionMessage(), grpc::StatusCode::INTERNAL, true});
                }
            })
            .IgnoreResult();
    }

    bool ShouldRetry() const override {
        // TODO(ivanzhukov): take into account cases when it shouldn't be retried (exceptions, network overload, etc.)
        return ShouldRetry_;
    }

    TString What() const override {
        return What_;
    }

    size_t TimesPerformed() const override {
        return TimesPerformed_;
    }

    std::unique_ptr<NActors::IEventBase> MakeReplyEvent() override {
        return {};
    }

    std::unique_ptr<NActors::IEventBase> MakeCompletionEvent() override {
        return std::make_unique<TPrivateEvents::TLeaderRespondedOnGetSlicesByHost>(
            std::move(Result_), std::move(LeaderAddress_), std::move(ReplyTo_));
    }

private:
    const TString& Service_;
    TString Host_;
    NSolomon::NSlicerClient::ISlicerClient* SlicerClient_;
    TString LeaderAddress_;
    NActors::TActorId ReplyTo_;

    NSlicerClient::TGetSlicesByHostResponseOrErrorPtr Result_;
    TString What_;
    size_t TimesPerformed_{0};
    bool ShouldRetry_{false};
};

} // namespace

std::shared_ptr<IRequestCtx> CreateForwardedGetSlicesByHostRequestCtx(
    const TString& service,
    TString host,  // NOLINT(performance-unnecessary-value-param): false positive
    NSolomon::NSlicerClient::ISlicerClient* slicerClient,
    TString leaderAddress,  // NOLINT(performance-unnecessary-value-param): false positive
    NActors::TActorId replyTo)
{
    return std::make_shared<TForwardedGetSlicesByHostRequestCtx>(
        service,
        std::move(host),
        std::move(slicerClient),
        std::move(leaderAddress),
        replyTo);
}

} // namespace NSolomon::NSlicer::NApi
