#include "request_ctx_load_info.h"

namespace NSolomon::NSlicer::NApi {
namespace {

class TLoadInfoRequestCtx: public IRequestCtx {
public:
    TLoadInfoRequestCtx(
            NLoadInfo::ILoadInfoClusterClientPtr clusterClient,
            NClusterMembership::TNodeEndpoint gatherInfoAbout)
        : ClusterClient_{std::move(clusterClient)}
        , GatherInfoAbout_{std::move(gatherInfoAbout)}
    {
        What_ = TStringBuilder() << "[load info about " << GatherInfoAbout_ << "]";
    }

private:
    NThreading::TFuture<void> PerformRequest() override {
        ++TimesPerformed_;

        auto request = ClusterClient_->Get(GatherInfoAbout_)->GetLoadInfo(GatherInfoAbout_);
        return HandleResult(request);
    }

    NThreading::TFuture<void> HandleResult(const NLoadInfo::TAsyncLoadInfoResponse& result) {
        return result
            .Subscribe([this, reqCtxPtr{shared_from_this()}](const NLoadInfo::TAsyncLoadInfoResponse& future) {
                try {
                    auto& valueOrError = future.GetValueSync();
                    if (valueOrError.Success()) {
                        // TODO(ivanzhukov): prevent the whole proto response from copying
                        Result_ = std::make_unique<NLoadInfo::TAsyncLoadInfoResponse::value_type>(valueOrError.Value());
                    } else {
                        ShouldRetry_ = true;
                        // TODO(ivanzhukov@): forward request to other nodes (as in SWIM)
                        Result_ = std::make_unique<NLoadInfo::TAsyncLoadInfoResponse::value_type>(valueOrError.Error());
                    }
                } catch (...) {
                    ShouldRetry_ = true;
                    Result_ = std::make_unique<NLoadInfo::TAsyncLoadInfoResponse::value_type>(
                        // TODO(ivanzhukov): both TApiCallError and an inner TGrpcStatus store a string message
                        //  -- use one type for errors instead
                        NSolomon::TApiCallError{
                            NGrpc::TGrpcStatus{"", grpc::StatusCode::INTERNAL, true},
                            CurrentExceptionMessage()});
                }
            })
            .IgnoreResult();
    }

    bool ShouldRetry() const override {
        // TODO(ivanzhukov): take into account cases when it shouldn't be retried (exceptions, network overload, etc.)
        return ShouldRetry_;
    }

    TString What() const override {
        return What_;
    }

    size_t TimesPerformed() const override {
        return TimesPerformed_;
    }

    std::unique_ptr<NActors::IEventBase> MakeReplyEvent() override {
        return std::make_unique<TPrivateEvents::TOneNodeProbingResult>(
            shared_from_this(),
            std::move(Result_),
            std::move(GatherInfoAbout_));
    }

    std::unique_ptr<NActors::IEventBase> MakeCompletionEvent() override {
        return std::make_unique<TPrivateEvents::TClusterProbingCompleted>();
    }

private:
    NLoadInfo::ILoadInfoClusterClientPtr ClusterClient_;
    NClusterMembership::TNodeEndpoint GatherInfoAbout_;

    std::unique_ptr<NLoadInfo::TAsyncLoadInfoResponse::value_type> Result_;
    TString What_;
    // TODO(ivanzhukov): check thread-safety
    size_t TimesPerformed_{0};
    bool ShouldRetry_{false};
};


} // namespace

std::shared_ptr<IRequestCtx> CreateLoadInfoRequestCtx(
    NLoadInfo::ILoadInfoClusterClientPtr clusterClient,
    NClusterMembership::TNodeEndpoint gatherInfoAbout)
{
    return std::make_shared<TLoadInfoRequestCtx>(std::move(clusterClient), std::move(gatherInfoAbout));
}

} // namespace NSolomon::NSlicer::NApi
