#include "assignment_manager.h"
#include "events.h"
#include "service_manager.h"

#include <solomon/libs/cpp/actors/config/log_component.pb.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>

#include <util/system/hostname.h>

namespace NSolomon::NSlicer {
namespace {

using namespace NActors;
using namespace yandex::monitoring::slicer;

IDistributedLockPtr CreateLock(const NYdb::TDriver& driver, const SlicerConfig::LockConfig& lockConf, TStringBuf serviceName) {
    auto path = lockConf.node_path();
    Y_ENSURE(!path.Empty(), "path to the lock cannot be empty");
    Y_VERIFY(!lockConf.semaphore_name_prefix().empty(), "semaphore_name_preifx cannot be empty");
    auto name = lockConf.semaphore_name_prefix() + "_" + serviceName;
    Y_ENSURE(!name.Empty(), "semaphore name cannot be empty");
    auto data = FQDNHostName();

    TYdbLockConfig config{
        .Driver = driver,
        .Path = path,
        .Name = name,
        .Data = data,
    };

    auto f = PrepareYdbLock(config);
    auto ok = f.Wait(TDuration::Minutes(1));

    Y_ENSURE(ok, "YDB prepare timed out");
    return CreateYdbLock(config);
}

struct TActorsInfo {
    TActorId AssignmentManager;
    TActorId ClusterMembership;
};

class TServiceManager: public TActorBootstrapped<TServiceManager> {
public:
    TServiceManager(
            const SlicerConfig& config,
            NYdb::TDriver ydbDriver,
            TClusterMembershipFactory clusterMembershipFactory,
            NSolomon::NSlicerClient::ISlicerClusterClientPtr slicerClients,
            NMonitoring::TMetricRegistry& registry,
            TString clientId,
            NDb::IAssignmentDaoPtr assnDao,
            std::shared_ptr<NDb::IServiceConfigDao> serviceDao)
        : Config_{config}
        , LockConfig_{Config_.leader_election_config().lock_config()}
        , YdbDriver_{std::move(ydbDriver)}
        , ClusterMembershipFactory_{std::move(clusterMembershipFactory)}
        , SlicerClients_{std::move(slicerClients)}
        , Registry_{registry}
        , ClientId_{std::move(clientId)}
        , AssnDao_{std::move(assnDao)}
        , ServiceDao_{std::move(serviceDao)}
    {
        for (const auto& service: config.known_services()) {
            KnownServices_.emplace(service);
        }
    }

    TActorsInfo& CreateNewAssignmentManager(const TString& service) {
        auto clusterMembershipActor = Register(ClusterMembershipFactory_(service, Config_, Registry_).Release());
        auto newActor = Register(NApi::CreateAssignmentManager(
            service,
            Config_,
            // FIXME(ivanzhukov): make it non-ingestor-specific (right now it stores ingestor addresses)
            clusterMembershipActor,
            CreateLock(YdbDriver_, LockConfig_, service),
            SlicerClients_,
            Registry_,
            ClientId_,
            AssnDao_,
            ServiceDao_).Release());

        auto& info = ServiceToActors_[service];
        info.AssignmentManager = newActor;
        info.ClusterMembership = clusterMembershipActor;

        return info;
    }

    void Bootstrap() {
        CreateNewAssignmentManager("ingestor");

        Become(&TThis::Main);
    }

    STATEFN(Main) {
        switch (ev->GetTypeRewrite()) {
            //gRpc
            hFunc(TAssignmentEvents::TGetAllAssignments, ForwardToAssignmentManager<TAssignmentEvents::TGetAllAssignments>);
            hFunc(TAssignmentEvents::TGetSlicesByHost, ForwardToAssignmentManager<TAssignmentEvents::TGetSlicesByHost>);
            hFunc(TAssignmentEvents::TMoveShardRequest, ForwardToAssignmentManager<TAssignmentEvents::TMoveShardRequest>);

            // http
            hFunc(TAssignmentEvents::TGetClusterStatus, ForwardToAssignmentManager<TAssignmentEvents::TGetClusterStatus>);
            hFunc(TAssignmentEvents::TGetHostInfo, ForwardToAssignmentManager<TAssignmentEvents::TGetHostInfo>);
            hFunc(TAssignmentEvents::TGetShardsHostInfo, ForwardToAssignmentManager<TAssignmentEvents::TGetShardsHostInfo>);
            hFunc(TAssignmentEvents::TValidateShardMoveParams, ForwardToAssignmentManager<TAssignmentEvents::TValidateShardMoveParams>);
            hFunc(TAssignmentEvents::TAssignUniformly, ForwardToAssignmentManager<TAssignmentEvents::TAssignUniformly>);
            hFunc(TAssignmentEvents::TRebalanceOnceByCount, ForwardToAssignmentManager<TAssignmentEvents::TRebalanceOnceByCount>);
            hFunc(TAssignmentEvents::TRebalanceOnceByCpu, ForwardToAssignmentManager<TAssignmentEvents::TRebalanceOnceByCpu>);
            hFunc(TAssignmentEvents::TRebalanceOnceByMemory, ForwardToAssignmentManager<TAssignmentEvents::TRebalanceOnceByMemory>);
            hFunc(TAssignmentEvents::TChangeServiceSettings, ForwardToAssignmentManager<TAssignmentEvents::TChangeServiceSettings>);
            hFunc(TAssignmentEvents::TScheduleRebalancing, ForwardToAssignmentManager<TAssignmentEvents::TScheduleRebalancing>);
            hFunc(TServiceManagerEvents::TGetServices, OnGetServices);
        }
    }

private:
    template <typename TReq>
    void ReplyToUnknownService(typename TReq::TPtr& evPtr) {
        TString msg = TStringBuilder() << "unkown service \"" << evPtr->Get()->Service << "\"";

        MON_WARN(ServiceManager, msg);

        // TODO(ivanzhukov):
//        using TResp = TReqToResp<TReq>::TResp;
//        Send(evPtr->Sender, new TResp{TResp::Response::FromError{"unkown service \"" << service << "\""}});
    }

    template <typename TReq>
    void ForwardToAssignmentManager(typename TReq::TPtr& evPtr) {
        TString service = evPtr->Get()->Service;
        if (service.empty()) {
            // FIXME(ivanzhukov): remove this fallback when multiple services are supported
            service = "ingestor";
        }

        TActorId assignmentManager;

        if (auto actorsIt = ServiceToActors_.find(service); actorsIt == ServiceToActors_.end()) {
            if constexpr (std::is_same_v<TReq, TAssignmentEvents::TGetSlicesByHost>) { // TODO: change to TRegister
                if (KnownServices_.contains(service)) {
                    assignmentManager = CreateNewAssignmentManager(service).AssignmentManager;
                } else {
                    ReplyToUnknownService<TReq>(evPtr);
                    return;
                }
            } else {
                ReplyToUnknownService<TReq>(evPtr);
                return;
            }
        } else {
            assignmentManager = actorsIt->second.AssignmentManager;
        }

        TActivationContext::Send(evPtr->Forward(assignmentManager));
    }

    void OnGetServices(const TServiceManagerEvents::TGetServices::TPtr& evPtr) {
        TVector<TString> services{::Reserve(ServiceToActors_.size())};
        for (const auto& [service, _]: ServiceToActors_) {
            services.emplace_back(service);
        }

        Send(evPtr->Sender, new TServiceManagerEvents::TGetServicesResponse{std::move(services)});
    }

private:
    const SlicerConfig& Config_;
    const SlicerConfig::LockConfig& LockConfig_;
    yandex::solomon::config::rpc::TGrpcClientConfig GrpcClientConfig_;
    NYdb::TDriver YdbDriver_;
    TClusterMembershipFactory ClusterMembershipFactory_;
    NSolomon::NSlicerClient::ISlicerClusterClientPtr SlicerClients_;
    NMonitoring::TMetricRegistry& Registry_;
    TString ClientId_;
    NDb::IAssignmentDaoPtr AssnDao_;
    std::shared_ptr<NDb::IServiceConfigDao> ServiceDao_;

    absl::flat_hash_map<TString, TActorsInfo> ServiceToActors_;
    // TODO(ivanzhukov): register new services via UI or smth
    absl::flat_hash_set<TString> KnownServices_;
};

} // namespace

std::unique_ptr<NActors::IActor> CreateServiceManager(
    const yandex::monitoring::slicer::SlicerConfig& config,
    NYdb::TDriver ydbDriver,
    TClusterMembershipFactory clusterMembershipFactory,
    NSolomon::NSlicerClient::ISlicerClusterClientPtr slicerClients,
    NMonitoring::TMetricRegistry& registry,
    TString clientId,
    NDb::IAssignmentDaoPtr assnDao,
    std::shared_ptr<NDb::IServiceConfigDao> serviceDao)
{
    return std::make_unique<TServiceManager>(
        config,
        std::move(ydbDriver),
        std::move(clusterMembershipFactory),
        std::move(slicerClients),
        registry,
        std::move(clientId),
        std::move(assnDao),
        std::move(serviceDao));
}

} // namespace NSolomon::NSlicer
