#pragma once

#include "balancer.h"

#include <solomon/services/slicer/lib/common/host_reported.h>
#include <solomon/services/slicer/lib/common/slices.h>
#include <solomon/services/slicer/lib/common/reassignment_type.h>

namespace NSolomon::NSlicer::NBalancer {

struct TNodeWithLoad {
    NClusterMembership::TNodeEndpoint Node;
    ui64 Budget{0};
    ui64 Load{0};
    double LoadNorm{0};

    TNodeWithLoad(NClusterMembership::TNodeEndpoint node, ui64 budget, ui64 load, double loadNorm)
            : Node{std::move(node)}
            , Budget{budget}
            , Load{load}
            , LoadNorm{loadNorm}
    {
    }
};

struct THottestAndColdest {
    TNodeWithLoad Hottest;
    TNodeWithLoad Coldest;
};

struct TMove {
    double Weight{0.0};
    TSliceWithLoad Slice;
};

using TSlicesWithShards = std::map<NApi::TSlice, std::map<NApi::TNumId, TLoadInfo>>;

TString ToJson(const TStringMap<NSolomon::NSlicer::NApi::TSlices>& hostToSlices, bool indented = false);

bool AreEqual(
        const TStringMap<NSolomon::NSlicer::NApi::TSlices>& hostToSlices1,
        const TStringMap<NSolomon::NSlicer::NApi::TSlices>& hostToSlices2);

bool AreEqual(
        const TStringMap<TSlicesWithShards>& hostToSlices1,
        const TStringMap<TSlicesWithShards>& hostToSlices2,
        bool ignoreSliceBounds = true);

struct TLoadDistributionDescription {
    double MeanLoad{0.0};
    double StandardDeviation{0.0};
};

class TSliceLoadBalancer {
public:
    TSliceLoadBalancer(
            const NDb::TServiceConfig& serviceSettings,
            const TStringMap<TLoadInfo>& hostsInfo,
            const TStringMap<NApi::TSlices>& hostSlices,
            const absl::flat_hash_map<NApi::TNumId, TLoadInfo>& shardsInfo,
            const NActors::TActorContext* actorContext = nullptr,
            const TString& logPrefix = {});

    void BalanceByCpu();
    void BalanceByMemory();
    [[nodiscard]] TStringMap<NApi::TSlices> BuildHostToSlicesMapping() const;
    [[nodiscard]] TAssignments BuildAssignments() const;
    [[nodiscard]] TStringMap<TSlicesWithShards> BuildHostToSlicesWithShardsMapping() const;
    const TBalanceStats& GetStatistics() const {
        return Stats_;
    }
    const THostToSliceSet& GetHostToSlices() const {
        return HostToSlices_;
    }
    // it is temporary method to ensure class invariant, until stabilization is done
    bool CorrectSliceLoadValues();

    /**
     * Only for unit tests
     */
     void MergeSlices(EReassignmentType type);
     void MoveSlices(EReassignmentType type);
     void SplitHotSlices(EReassignmentType type);
     bool CheckSliceLoadValues() const;

private:
    void BuildHostToSlices(const TStringMap<NApi::TSlices>& hostSlices);

    absl::flat_hash_map<NApi::TSlice, TLoadInfo, THash<NApi::TSlice>> CalcSlicesLoad(
            const TStringMap<NApi::TSlices>& hostSlices) const;

    static TVector<NApi::TNumId> BuildSortedNumIds(const absl::flat_hash_map<NApi::TNumId, TLoadInfo>& shardsInfo);

    static TLoadInfo CalcTotalLoad(const absl::flat_hash_map<NApi::TNumId, TLoadInfo>& shardsInfo);

    static TLoadInfo CalcTotalBudget(const TStringMap<TLoadInfo>& hostsInfo);

    template <EReassignmentType RType>
    static ui64 GetLoad(const TLoadInfo& loadInfo);

    template <EReassignmentType RType>
    void DoBalance();

    template <EReassignmentType RType>
    void MergeAdjacentColdSlices();

    TVector<TSliceWithLoadAndHost> BuildSlicesWithLoad() const;

    TStringMap<TLoadInfo> BuildHostToLoad() const;

    template <EReassignmentType RType>
    double CalcMaxHostLoad(const TStringMap<TLoadInfo>& hostToLoad) const;

    template <EReassignmentType RType>
    void MoveSlices();

    template <EReassignmentType RType>
    void FillNodesWithLoads(absl::flat_hash_set<TString>& filtered, TVector<TNodeWithLoad>& nodes) const;

    template <EReassignmentType RType>
    TVector<double> GetNodeLoads() const;

    THottestAndColdest DetectHottestAndColdestNodes(
            ui64& totalLoad,
            ui64& totalBudget,
            double& meanLoad,
            absl::flat_hash_set<TString>& filtered,
            TVector<TNodeWithLoad>& nodes) const;

    template <EReassignmentType RType>
    std::optional<TMove> FindBestMove(
            const TNodeWithLoad& hottestNode,
            const TNodeWithLoad& coldestNode,
            double meanNodeLoad) const;

    template <EReassignmentType RType>
    void SplitHotSlices();

    template <EReassignmentType RType>
    bool SplitHotSlicesStep(
            TVector<TSliceWithLoadAndHost>& slices,
            absl::flat_hash_set<NApi::TSlice, THash<NApi::TSlice>>& unsplittable,
            ui64& totalLoad,
            size_t& slicesBudget);

    template <EReassignmentType RType>
    size_t GetSliceShards(const NApi::TSlice& slice, TVector<NApi::TNumId>& sliceNumIds) const;

    TVector<NApi::TNumId> GetSliceShards(const NApi::TSlice& slice) const;

    template <EReassignmentType RType>
    TSliceWithLoadAndHost SplitHotSlice(
            TSliceWithLoadAndHost& slice,
            const TVector<NApi::TNumId>& sliceNumIds,
            ui64 meanSliceLoad) const;

    template <EReassignmentType RType>
    void DescribeDistribution();

private:
    const NActors::TActorContext* ActorContext_{nullptr};
    const TString LogPrefix_;
    const NDb::TServiceConfig& ServiceSettings_;
    const TStringMap<TLoadInfo>& HostsInfo_;
    const absl::flat_hash_map<NApi::TNumId, TLoadInfo>& ShardsInfo_;
    const TVector<NApi::TNumId> SortedNumIds_;
    const TLoadInfo TotalLoad_;
    const TLoadInfo TotalBudget_;
    THostToSliceSet HostToSlices_;
    TBalanceStats Stats_{};
};

} // namespace NSolomon::NSlicer::NBalancer
