#include <solomon/services/slicer/lib/balancer/balancer.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon;
using namespace NSolomon::NSlicer;

class TBalancerTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorRuntime = TTestActorRuntime::CreateInited(1, false, false);
        Registry = std::make_shared<NMonitoring::TMetricRegistry>();
        EdgeId = ActorRuntime->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime.Reset();
    }

    THolder<NSolomon::TTestActorRuntime> ActorRuntime;
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry;
    TActorId EdgeId;
};

TEST_F(TBalancerTest, Dummy) {
    ActorRuntime->Register(CreateBalancer(
        "ingestor",
        EdgeId,
        EReassignmentType::ByCpu,
        NSolomon::NClusterMembership::TClusterMembershipState{},
        TStringMap<NSolomon::NSlicer::NApi::TSlices>{},
        THostReportedInfo{},
        NDb::TServiceConfig{}).release());
    ActorRuntime->WaitForBootstrap();

    auto ev = ActorRuntime->GrabEdgeEvent<TBalancerEvents::TBalanceResult>(EdgeId);

    ASSERT_TRUE(ev);
    EXPECT_TRUE(ev->Get()->Assignments.empty());
    auto& stats = ev->Get()->Stats;
    EXPECT_EQ(stats.MoveKeyChurn, 0u);
    EXPECT_EQ(stats.MoveIterations, 0u);
    EXPECT_EQ(stats.NumOfMovedSlices, 0u);
    EXPECT_EQ(stats.MergeKeyChurn, 0u);
    EXPECT_EQ(stats.MergeIterations, 0u);
    EXPECT_EQ(stats.NumOfMergedSlices, 0u);
    EXPECT_EQ(stats.MergeStatus, EMergeStatus::Unknown);
}

TEST_F(TBalancerTest, RebalanceByCount) {
    NSolomon::NClusterMembership::TClusterMembershipState clusterMembership{
        {"alive_1", {.State = NClusterMembership::ENodeState::Alive}},
        {"alive_2", {.State = NClusterMembership::ENodeState::Alive}},
    };
    TStringMap<NSolomon::NSlicer::NApi::TSlices> hostToSlices{
        {"alive_1", { {1, Max<NApi::TNumId>()} }},
        {"alive_2", {}},
    };

    ActorRuntime->Register(CreateBalancer(
            "ingestor",
            EdgeId,
            EReassignmentType::ByCount,
            std::move(clusterMembership),
            std::move(hostToSlices),
            THostReportedInfo{},
            NDb::TServiceConfig{}).release());
    ActorRuntime->WaitForBootstrap();

    auto ev = ActorRuntime->GrabEdgeEvent<TBalancerEvents::TBalanceResult>(EdgeId);

    auto firstSlice = NApi::TSlice{1, Max<NApi::TNumId>() / 2};
    auto secondSlice = NApi::TSlice{Max<NApi::TNumId>() / 2 + 1, Max<NApi::TNumId>()};

    ASSERT_TRUE(ev);
    auto& assn = ev->Get()->Assignments;
    ASSERT_EQ(assn.size(), 2ul);
    EXPECT_EQ(assn.begin()->first, firstSlice);
    EXPECT_EQ((++assn.begin())->first, secondSlice);

    auto& hts = ev->Get()->HostToSlices;
    ASSERT_EQ(hts.size(), 2ul);
    EXPECT_EQ(hts["alive_1"sv].size(), 1ul);
    EXPECT_EQ(*hts["alive_1"sv].begin(), secondSlice);
    EXPECT_EQ(hts["alive_2"sv].size(), 1ul);
    EXPECT_EQ(*hts["alive_2"sv].begin(), firstSlice);
}
