#include "assignments.h"

#include <solomon/libs/cpp/slices/operations.h>

namespace NSolomon::NSlicer {

using namespace NApi;
using namespace NClusterMembership;

namespace {

template <typename TStream>
void Out(TStream& s, const TAssignments& assn) {
    s << '{';

    size_t i = 0;
    for (const auto& as: assn) {
        if (i++ > 0) {
            s << ", ";
        }
        s << '(' << as.first << " -> " << as.second << ')';
    }

    s << '}';
}

template <typename TStream>
void Out(TStream& s, const TAssignmentsDiff& diff) {
    s << "add: " << diff.Added << "; rm: [";

    size_t i = 0;
    for (const auto& key: diff.Removed) {
        if (i++ > 0) {
            s << ", ";
        }
        s << key;
    }

    s << "]";
}

bool CheckAssignmentsForValidity(const TAssignments& assn, bool isSafe) {
    ui64 keyspaceSize = 0;
    size_t i = 0;

    for (auto it = assn.begin(); it != assn.end(); ++it, ++i) {
        const auto& slice = it->first;
        keyspaceSize += slice.Size();

        if (slice.Start > slice.End) {
            auto& s = slice;

            if (isSafe) {
                return false;
            }

            Cerr << "invalid slice: " << s << Endl;
            Y_VERIFY(false, "slice.Start > slice.End");
        }

        if (i > 0) {
            auto prev = it;
            --prev;

            if (slice.Start == prev->first.End + 1) {
                continue;
            }

            if (isSafe) {
                return false;
            }

            Y_VERIFY(false, "keyspace contains holes");
        }
    }

    if (keyspaceSize != Max<TNumId>()) {
        if (isSafe) {
            return false;
        }

        Y_VERIFY(false, "keyspace is too small");
    }

    return true;
}

} // namespace

TAssignmentsDiff DiffAssignments(const TAssignments& oldAssn, const TAssignments& newAssn) {
    TAssignmentsDiff diff;

    auto oldIt = oldAssn.begin();
    auto newIt = newAssn.begin();

    auto removeOld = [&]() {
        diff.Removed.emplace_back(oldIt->first);
        ++oldIt;
    };
    auto addNew = [&]() {
        diff.Added.emplace(newIt->first, newIt->second);
        ++newIt;
    };

    while (oldIt != oldAssn.end() && newIt != newAssn.end()) {
        const auto& oldSlice = oldIt->first;
        const auto& newSlice = newIt->first;

        if (oldSlice.Start == newSlice.Start) {
            if (oldSlice.End == newSlice.End) {
                if (oldIt->second == newIt->second) {
                    ++oldIt;
                    ++newIt;
                } else {
                    removeOld();
                    addNew();
                }

                continue;
            }

            if (oldSlice.End < newSlice.End) {
                removeOld();
            } else {
                addNew();
            }
        } else if (oldSlice.Start < newSlice.Start) {
            removeOld();
        } else {
            addNew();
        }
    }

    while (oldIt != oldAssn.end()) {
        removeOld();
    }

    while (newIt != newAssn.end()) {
        addNew();
    }

    return diff;
}

TAssignments ConstructAssignmentsFromHostToSlices(const TStringMap<NApi::TSlices>& hostToSlices) {
    TAssignments assignments;

    for (const auto& [host, slices]: hostToSlices) {
        for (const auto& slice: slices) {
            auto it = assignments.try_emplace(slice, THosts{}).first;
            it->second.emplace_back(host);
        }
    }

    return assignments;
}

TStringMap<NApi::TSlices> ConstructHostToSlicesFromAssignments(TAssignments& assn) {
    TStringMap<NApi::TSlices> hostToSlices;

    for (const auto& [slice, hosts]: assn) {
        for (const auto& host: hosts) {
            hostToSlices[host].emplace(slice);
        }
    }

    return hostToSlices;
}

void MoveSlicesFromDeadNodes(
        const TVector<TStringBuf>& deadNodes,
        const TVector<TStringBuf>& assignableNodes,
        TStringMap<TSlices>& hostToSlices)
{
    TSlices slicesToMoveFromDeadNodes;

    for (auto& node: deadNodes) {
        auto nh = hostToSlices.extract(node);
        if (nh.empty()) {
            continue;
        }

        for (auto& slice: nh.mapped()) {
            slicesToMoveFromDeadNodes.emplace(slice);
        }
    }

    if (slicesToMoveFromDeadNodes.empty()) {
        return;
    }

    // TODO(ivanzhukov): when the balancing algorithm is implemented, stop scatter slices evenly -- just rebalance by weights.
    //  Maybe it will be more convenient to create a sentinel or virtual node with zero cpu/mem/net capacity,
    //  assign all slices to it and then rebalance all nodes according to the algo
    auto newSlices = RearrangeSlicesIntoEvenParts(slicesToMoveFromDeadNodes, static_cast<ui32>(assignableNodes.size()));
    auto newSlicesIt = newSlices.begin();

    for (auto& node: assignableNodes) {
        auto& hostSlices = hostToSlices[node];

        for (auto slice: *newSlicesIt) {
            bool isNew = hostSlices.emplace(slice).second;
            Y_ENSURE(isNew, "the moved slice " << slice << " was already assigned to a host " << node);
        }

        if (++newSlicesIt == newSlices.end()) {
            break;
        }
    }
}

void RebalanceKeysUniformly(const TVector<TStringBuf>& assignableNodes, TStringMap<TSlices>& hostToSlices) {
    if (assignableNodes.size() <= 1) {
        return;
    }

    ui32 keySpaceSize = 0;
    for (const auto& host: assignableNodes) {
        keySpaceSize += SlicesSize(hostToSlices[host]);
    }

    ui32 targetNumOfKeys = keySpaceSize / assignableNodes.size();
    ui32 numOfKeysLeft = keySpaceSize % assignableNodes.size();
    TSlices slicesToMove;
    TVector<TStringBuf> starvingNodes;

    for (const auto& host: assignableNodes) {
        auto& slices = hostToSlices[host];
        ui32 slicesSize = SlicesSize(slices);
        ui32 targetSize = targetNumOfKeys;

        // In case of keySpaceSize not divisible by a number of nodes, there will be legitimate nodes with "extra" slices.
        if (slicesSize <= targetSize) {
            starvingNodes.emplace_back(host);
            continue;
        } else if (numOfKeysLeft > 0) {
            --numOfKeysLeft;
            ++targetSize;

            if (slicesSize == targetSize) {
                continue;
            }
        }

        ui32 numOfKeysToMove = slicesSize - targetSize;
        ExtractSlicesOfTotalSize(slices, slices.begin(), numOfKeysToMove, slicesToMove);
    }

    if (slicesToMove.empty()) {
        return;
    }

    for (auto& host: starvingNodes) {
        if (slicesToMove.empty()) {
            break;
        }

        auto& slices = hostToSlices[host];

        ui32 targetSize = targetNumOfKeys + (numOfKeysLeft > 0);
        ui32 numOfKeysToMove = targetSize - SlicesSize(slices);
        if (numOfKeysLeft) { --numOfKeysLeft; }

        ExtractSlicesOfTotalSize(slicesToMove, slicesToMove.begin(), numOfKeysToMove, slices);
    }

    Y_ENSURE(slicesToMove.empty(), "slices that need to be moved were left");
}

void CheckAssignmentsForValidity(const TAssignments& assn) {
    CheckAssignmentsForValidity(assn, false);
}

bool CheckAssignmentsForValiditySafe(const TAssignments& assn) {
    return CheckAssignmentsForValidity(assn, true);
}

} // namespace NSolomon::NSlicer

IOutputStream& operator<<(IOutputStream& os, const NSolomon::NSlicer::TAssignments& assn) {
    NSolomon::NSlicer::Out(os, assn);
    return os;
}

std::ostream& operator<<(std::ostream& os, const NSolomon::NSlicer::TAssignments& assn) {
    NSolomon::NSlicer::Out(os, assn);
    return os;
}

IOutputStream& operator<<(IOutputStream& os, const NSolomon::NSlicer::TAssignmentsDiff& diff) {
    NSolomon::NSlicer::Out(os, diff);
    return os;
}

std::ostream& operator<<(std::ostream& os, const NSolomon::NSlicer::TAssignmentsDiff& diff) {
    NSolomon::NSlicer::Out(os, diff);
    return os;
}
