#include <solomon/services/slicer/lib/common/assignments.h>

#include <solomon/libs/cpp/slices/operations.h>
#include <solomon/libs/cpp/string_map/string_map.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NClusterMembership;
using namespace NSolomon::NSlicer::NApi;
using namespace NSolomon::NSlicer;
using namespace NSolomon;
using namespace testing;

TEST(AssignmentsDiff, EmptyAssn) {
    TAssignments oldAssn;
    TAssignments newAssn;

    auto diff = DiffAssignments(oldAssn, newAssn);

    EXPECT_TRUE(diff.Added.empty());
    EXPECT_TRUE(diff.Removed.empty());
}

TEST(AssignmentsDiff, EmptyNewAssn) {
    TAssignments oldAssn{
        { {1, 2}, {} },
        { {3, 4}, {} },
    };
    TAssignments newAssn;

    auto diff = DiffAssignments(oldAssn, newAssn);

    EXPECT_TRUE(diff.Added.empty());
    EXPECT_EQ(oldAssn.size(), diff.Removed.size());

    auto oldIt = oldAssn.begin();
    auto slice = diff.Removed[0];
    EXPECT_EQ(slice.Start, oldIt->first.Start);
    EXPECT_EQ(slice.End, oldIt->first.End);

    ++oldIt;
    slice = diff.Removed[1];
    EXPECT_EQ(slice.Start, oldIt->first.Start);
    EXPECT_EQ(slice.End, oldIt->first.End);

    ++oldIt;
    EXPECT_EQ(oldIt, oldAssn.end());
}

TEST(AssignmentsDiff, EmptyOldAssn) {
    TAssignments oldAssn;
    TAssignments newAssn{
        { {1, 2}, {} },
        { {3, 4}, {} },
    };

    auto diff = DiffAssignments(oldAssn, newAssn);

    EXPECT_TRUE(diff.Removed.empty());
    EXPECT_EQ(newAssn.size(), diff.Added.size());

    auto newIt = newAssn.begin();
    auto addedIt = diff.Added.begin();
    EXPECT_EQ(addedIt->first.Start, newIt->first.Start);
    EXPECT_EQ(addedIt->first.End, newIt->first.End);

    ++newIt;
    ++addedIt;
    EXPECT_EQ(addedIt->first.Start, newIt->first.Start);
    EXPECT_EQ(addedIt->first.End, newIt->first.End);

    ++newIt;
    ++addedIt;
    EXPECT_EQ(newIt, newAssn.end());
    EXPECT_EQ(addedIt, diff.Added.end());
}

TEST(AssignmentsDiff, Equal) {
    TAssignments oldAssn{
        { {1, 2}, {} },
        { {3, 4}, {} },
    };
    TAssignments newAssn{
        { {1, 2}, {} },
        { {3, 4}, {} },
    };

    auto diff = DiffAssignments(oldAssn, newAssn);

    EXPECT_TRUE(diff.Removed.empty());
    EXPECT_TRUE(diff.Added.empty());
}

TEST(AssignmentsDiff, AllCases) {
    TAssignments oldAssn{
        { {1, 9}, {} },
        { {10, 11}, {} },
        { {12, 13}, {} },
        { {14, 15}, {} },
        { {16, 18}, {} },
        { {19, 20}, {} },
        { {21, 26}, {} },
    };
    TAssignments newAssn{
        { {1, 2}, {} },
        { {3, 6}, {} },
        { {7, 10}, {} },
        { {11, 11}, {} },
        { {12, 13}, {} },
        { {14, 18}, {} },
        { {19, 23}, {} },
        { {24, 26}, {} },
    };

    auto diff = DiffAssignments(oldAssn, newAssn);

    TVector<TSlice> expectedRemoved = {
        {1, 9},
        {10, 11},
        {14, 15},
        {16, 18},
        {19, 20},
        {21, 26},
    };
    ASSERT_EQ(expectedRemoved.size(), diff.Removed.size());

    for (auto expIt = expectedRemoved.begin(), removedIt = diff.Removed.begin(); removedIt != diff.Removed.end(); ++expIt, ++removedIt) {
        EXPECT_EQ(*expIt, *removedIt) << "expected: " << *expIt << "; got: " << *removedIt;
    }

    TAssignments expectedAdded{
        { {1, 2}, {} },
        { {3, 6}, {} },
        { {7, 10}, {} },
        { {11, 11}, {} },
        { {14, 18}, {} },
        { {19, 23}, {} },
        { {24, 26}, {} },
    };
    ASSERT_EQ(expectedAdded.size(), diff.Added.size());

    for (auto expIt = expectedAdded.begin(), addedIt = diff.Added.begin(); addedIt != diff.Added.end(); ++expIt, ++addedIt) {
        EXPECT_EQ(*expIt, *addedIt) << "expected: " << expIt->first << "; got: " << addedIt->first;
    }
}

TEST(AssignmentsDiff, OnlyHostChanges) {
    TAssignments oldAssn{
        { {1, 9}, {"host1"} },
    };
    TAssignments newAssn{
        { {1, 9}, {"host2"} },
    };

    auto diff = DiffAssignments(oldAssn, newAssn);
    EXPECT_EQ(diff.Removed.size(), 1ul);
    EXPECT_EQ(diff.Added.size(), 1ul);

    EXPECT_EQ(*diff.Removed.begin(), oldAssn.begin()->first);
    EXPECT_EQ(*diff.Added.begin(), *newAssn.begin());
}

TEST(ConstructAssignmentsFromHostToSlices, Test) {
    {
        TStringMap<TSlices> hostToSlices;
        auto assignments = ConstructAssignmentsFromHostToSlices(hostToSlices);

        EXPECT_EQ(assignments.size(), 0ul);
    }

    {
        TStringMap<TSlices> hostToSlices;
        TSlices slices{ {1, 1} };
        hostToSlices.emplace("host_1", slices);

        auto assignments = ConstructAssignmentsFromHostToSlices(hostToSlices);
        EXPECT_EQ(assignments.size(), 1ul);

        auto it = assignments.begin();
        EXPECT_EQ(it->first, *slices.begin());
        EXPECT_EQ(it->second, TVector<TNodeEndpoint>{"host_1"});
    }

    {
        TStringMap<TSlices> hostToSlices;
        hostToSlices.emplace("host_1", TSlices{ {1, 1} });
        hostToSlices.emplace("host_2", TSlices{ {10, 20}, {30, 40} });
        hostToSlices.emplace("host_3", TSlices{ {10, 20}, {30, 40} });
        hostToSlices.emplace("host_4", TSlices{ {51, 69} });

        auto assignments = ConstructAssignmentsFromHostToSlices(hostToSlices);
        EXPECT_EQ(assignments.size(), 4ul);

        auto it = assignments.begin();
        EXPECT_EQ(it->first, (TSlice{1, 1}));
        EXPECT_EQ(it->second, TVector<TNodeEndpoint>{"host_1"});

        ++it;
        EXPECT_EQ(it->first, (TSlice{10, 20}));
        EXPECT_THAT(it->second, UnorderedElementsAre(TString{"host_2"}, TString{"host_3"}));

        ++it;
        EXPECT_EQ(it->first, (TSlice{30, 40}));
        EXPECT_THAT(it->second, UnorderedElementsAre(TString{"host_2"}, TString{"host_3"}));

        ++it;
        EXPECT_EQ(it->first, (TSlice{51, 69}));
        EXPECT_EQ(it->second, (TVector<TNodeEndpoint>{ "host_4" }));
    }
}

TEST(ConstructHostToSlicesFromAssignments, Test) {
    {
        TAssignments assn;
        assn.emplace(TSlice{1, 10}, THosts{ "node_1" });
        assn.emplace(TSlice{11, 20}, THosts{ "node_2" });

        auto hostToSlices = ConstructHostToSlicesFromAssignments(assn);

        ASSERT_EQ(hostToSlices.size(), 2ul);
        EXPECT_EQ(hostToSlices["node_1"sv], (TSlices{ {1, 10} }));
        EXPECT_EQ(hostToSlices["node_2"sv], (TSlices{ {11, 20} }));
    }

    {
        TAssignments assn;
        assn.emplace(TSlice{1, 10}, THosts{ "node_1" });
        assn.emplace(TSlice{11, 20}, THosts{ "node_1" });
        assn.emplace(TSlice{21, 30}, THosts{ "node_1" });
        assn.emplace(TSlice{98, 99}, THosts{ "node_2" });
        assn.emplace(TSlice{100, 100}, THosts{ "node_3" });

        auto hostToSlices = ConstructHostToSlicesFromAssignments(assn);

        ASSERT_EQ(hostToSlices.size(), 3ul);
        EXPECT_EQ(hostToSlices["node_1"sv], (TSlices{ {1, 10}, {11, 20}, {21, 30} }));
        EXPECT_EQ(hostToSlices["node_2"sv], (TSlices{ {98, 99} }));
        EXPECT_EQ(hostToSlices["node_3"sv], (TSlices{ {100, 100} }));
    }
}

TEST(MoveSlicesFromDeadNodes, Test) {
    {
        TStringMap<TSlices> hostToSlices;
        hostToSlices.emplace("dead_1", TSlices{ {1, 10} });
        hostToSlices.emplace("alive_1", TSlices{ {20, 30} });
        hostToSlices.emplace("alive_2", TSlices{ {40, 50} });
        hostToSlices.emplace("alive_3", TSlices{ {60, 70} });

        TVector<TStringBuf> deadNodes{"dead_1"};
        TVector<TStringBuf> aliveNodes{"alive_1", "alive_2", "alive_3"};

        MoveSlicesFromDeadNodes(deadNodes, aliveNodes, hostToSlices);

        ASSERT_EQ(hostToSlices.size(), 3ul);
        EXPECT_FALSE(hostToSlices.contains("dead_1"sv));

        EXPECT_EQ(hostToSlices["alive_1"sv], (TSlices{
            {1, 4},
            {20, 30},
        }));
        EXPECT_EQ(hostToSlices["alive_2"sv], (TSlices{
            {5, 7},
            {40, 50},
        }));
        EXPECT_EQ(hostToSlices["alive_3"sv], (TSlices{
            {8, 10},
            {60, 70},
        }));
    }

    {
        TStringMap<TSlices> hostToSlices;
        hostToSlices.emplace("dead_1", TSlices{ {1, 1} });
        hostToSlices.emplace("alive_1", TSlices{ {20, 30} });
        hostToSlices.emplace("alive_2", TSlices{ {40, 50} });
        hostToSlices.emplace("alive_3", TSlices{ {60, 70} });

        TVector<TStringBuf> deadNodes{"dead_1"};
        TVector<TStringBuf> aliveNodes{"alive_1", "alive_2", "alive_3"};

        MoveSlicesFromDeadNodes(deadNodes, aliveNodes, hostToSlices);

        ASSERT_EQ(hostToSlices.size(), 3ul);
        EXPECT_FALSE(hostToSlices.contains("dead_1"sv));

        EXPECT_EQ(hostToSlices["alive_1"sv], (TSlices{
                {1, 1},
                {20, 30},
        }));
        EXPECT_EQ(hostToSlices["alive_2"sv], (TSlices{
                {40, 50},
        }));
        EXPECT_EQ(hostToSlices["alive_3"sv], (TSlices{
                {60, 70},
        }));
    }
}

void CheckSlicesAfterRebalancing(TStringMap<TSlices>& hostToSlices, const TVector<TStringBuf>& assignableNodes) {
    ui32 totalSizeBefore = 0;
    for (auto& [_, slices]: hostToSlices) {
        totalSizeBefore += SlicesSize(slices);
    }

    RebalanceKeysUniformly(assignableNodes, hostToSlices);

    auto hasRightSize = [&](auto slices) {
        auto size = SlicesSize(slices);
        ui32 targetSize = totalSizeBefore / assignableNodes.size();

        return size == targetSize || size == targetSize + 1;
    };

    for (auto& host: assignableNodes) {
        EXPECT_TRUE(hasRightSize(hostToSlices[host])) << host << "'s slices: " << hostToSlices[host];
    }

    ui32 totalSizeAfter = 0;
    for (auto& [_, slices]: hostToSlices) {
        totalSizeAfter += SlicesSize(slices);
    }
    EXPECT_EQ(totalSizeBefore, totalSizeAfter);
}

TEST(RebalanceKeysUniformlyTest, Divisible) {;
    TStringMap<TSlices> hostToSlices{
        {"node_1", { {1, 9} }},
        {"node_2", { {11, 12} }},
        {"node_3", { {13, 19} }},
    };
    TVector<TStringBuf> assignableNodes{
        "node_1"sv, "node_2"sv, "node_3"sv,
    };

    CheckSlicesAfterRebalancing(hostToSlices, assignableNodes);
}

TEST(RebalanceKeysUniformlyTest, NotDivisible) {
    TStringMap<TSlices> hostToSlices{
            {"node_1", { {1, 10} }},
            {"node_2", { {11, 13} }},
            {"node_3", { {14, 20} }},
    };
    TVector<TStringBuf> assignableNodes{
            "node_1"sv, "node_2"sv, "node_3"sv,
    };

    CheckSlicesAfterRebalancing(hostToSlices, assignableNodes);
}

TEST(RebalanceKeysUniformlyTest, OneNodeWithAlmostEveryKey) {
    TStringMap<TSlices> hostToSlices{
            {"node_1", { {1, 12} }},
            {"node_2", { {13, 13} }},
            {"node_3", { {14, 14} }},
    };
    TVector<TStringBuf> assignableNodes{
            "node_1"sv, "node_2"sv, "node_3"sv,
    };

    CheckSlicesAfterRebalancing(hostToSlices, assignableNodes);
}

TEST(RebalanceKeysUniformlyTest, OnlyOneNode) {
    TStringMap<TSlices> hostToSlices{
            {"node_1", { {1, Max<TNumId>()} }},
    };
    TVector<TStringBuf> assignableNodes{
            "node_1"sv,
    };

    CheckSlicesAfterRebalancing(hostToSlices, assignableNodes);
    EXPECT_EQ(hostToSlices["node_1"sv], (TSlices{ {1, Max<TNumId>()} }));
}

TEST(RebalanceKeysUniformlyTest, AlreadyUniformlyDistributed_1) {
    TStringMap<TSlices> hostToSlices{
            {"node_1", { {1, 3} }},
            {"node_2", { {4, 6} }},
    };
    TVector<TStringBuf> assignableNodes{
            "node_1"sv, "node_2"sv,
    };

    CheckSlicesAfterRebalancing(hostToSlices, assignableNodes);
    EXPECT_EQ(hostToSlices["node_1"sv], (TSlices{ {1, 3} }));
    EXPECT_EQ(hostToSlices["node_2"sv], (TSlices{ {4, 6} }));
}

TEST(RebalanceKeysUniformlyTest, AlreadyUniformlyDistributed_2) {
    TStringMap<TSlices> hostToSlices{
            {"node_1", { {1, 3} }},
            {"node_2", { {4, 7} }},
    };
    TVector<TStringBuf> assignableNodes{
            "node_1"sv, "node_2"sv,
    };

    CheckSlicesAfterRebalancing(hostToSlices, assignableNodes);
    EXPECT_EQ(hostToSlices["node_1"sv], (TSlices{ {1, 3} }));
    EXPECT_EQ(hostToSlices["node_2"sv], (TSlices{ {4, 7} }));
}

TEST(RebalanceKeysUniformlyTest, AlreadyUniformlyDistributed_3) {
    TStringMap<TSlices> hostToSlices{
            {"node_1", { {1, 4} }},
            {"node_2", { {5, 7} }},
    };
    TVector<TStringBuf> assignableNodes{
            "node_1"sv, "node_2"sv,
    };

    CheckSlicesAfterRebalancing(hostToSlices, assignableNodes);
    EXPECT_EQ(hostToSlices["node_1"sv], (TSlices{ {1, 4} }));
    EXPECT_EQ(hostToSlices["node_2"sv], (TSlices{ {5, 7} }));
}

TEST(WholePipeline, Test) {
    TAssignments assnBefore;
    assnBefore.emplace(TSlice{ 1, 10}, THosts{ "node_1" });
    assnBefore.emplace(TSlice{11, 20}, THosts{ "node_2" });
    assnBefore.emplace(TSlice{21, 30}, THosts{ "node_3" });
    assnBefore.emplace(TSlice{31, 40}, THosts{ "node_4" });
    auto hostToSlices = ConstructHostToSlicesFromAssignments(assnBefore);

    TVector<TStringBuf> deadNodes{"node_1"sv};
    TVector<TStringBuf> assignableNodes{"node_2"sv, "node_3"sv, "node_4"sv};

    MoveSlicesFromDeadNodes(deadNodes, assignableNodes, hostToSlices);

    ASSERT_EQ(hostToSlices.size(), 3ul);
    EXPECT_EQ(hostToSlices["node_2"sv], (TSlices{ {11, 20}, {1, 4} }));
    EXPECT_EQ(hostToSlices["node_3"sv], (TSlices{ {21, 30}, {5, 7} }));
    EXPECT_EQ(hostToSlices["node_4"sv], (TSlices{ {31, 40}, {8, 10} }));

    // should take no effect in this case, 'cause keys are already uniformly distributed
    RebalanceKeysUniformly(assignableNodes, hostToSlices);

    EXPECT_EQ(hostToSlices["node_2"sv], (TSlices{ {11, 20}, {1, 4} }));
    EXPECT_EQ(hostToSlices["node_3"sv], (TSlices{ {21, 30}, {5, 7} }));
    EXPECT_EQ(hostToSlices["node_4"sv], (TSlices{ {31, 40}, {8, 10} }));

    TAssignments assnAfter = ConstructAssignmentsFromHostToSlices(hostToSlices);
    TAssignmentsDiff diff = DiffAssignments(assnBefore, assnAfter);

    ASSERT_EQ(diff.Removed.size(), 1ul);
    EXPECT_EQ(diff.Removed[0], (TSlice{1, 10}));
    ASSERT_EQ(diff.Added.size(), 3ul);

    EXPECT_EQ(diff.Added, (TAssignments{
        { {1,  4}, THosts{"node_2"}, },
        { {5,  7}, THosts{"node_3"}, },
        { {8, 10}, THosts{"node_4"}, },
    }));
}

TEST(Validity, CheckAssignmentsForValidity) {
    {
        // keyspace is too small
        TAssignments assn;
        assn.emplace(TSlice{1, 1}, THosts{ "node_1" });

        ASSERT_FALSE(CheckAssignmentsForValiditySafe(assn));
    }

    {
        // contains holes
        TAssignments assn;
        assn.emplace(TSlice{1, 10}, THosts{ "node_1" });
        assn.emplace(TSlice{31, 40}, THosts{ "node_2" });

        ASSERT_FALSE(CheckAssignmentsForValiditySafe(assn));
    }

    {
        TAssignments assn;
        assn.emplace(TSlice{1, 1431655765}, THosts{ "node_1" });
        assn.emplace(TSlice{1431655766, 2863311530}, THosts{ "node_2" });
        assn.emplace(TSlice{2863311531, Max<TNumId>()}, THosts{ "node_3" });

        ASSERT_TRUE(CheckAssignmentsForValiditySafe(assn));
    }
}
