#include "config.h"

#include <solomon/libs/cpp/config/units.h>

#include <library/cpp/protobuf/util/pb_io.h>

#include <util/generic/hash_set.h>

#define SLCR_VALIDATE(condition, message) Y_ENSURE_EX(condition, TConfigValidationException() << message)

using yandex::monitoring::slicer::SlicerConfig;

namespace NSolomon::NSlicer {

SlicerConfig ParseTextProto(const TString& filePath) {
    SlicerConfig config;
    static auto options = EParseFromTextFormatOption::AllowUnknownField;

    try {
        ParseFromTextFormat(filePath, config, options);
    } catch (...) {
        throw TConfigParseException() << CurrentExceptionMessage() << Endl;
    }

    return config;
}

// TODO(ivanzhukov@): move to a common place for all services
void Validate(const SlicerConfig& config) {
    THashSet<TStringBuf> ThreadPoolNames;

    // ActorSystem
    {
        SLCR_VALIDATE(config.Hasactor_system(), "actor_system{} section cannot be empty");
        const auto& asConfig = config.actor_system();

        // ActorSystem/Executors
        for (const auto& e: asConfig.executors()) {
            bool inserted = ThreadPoolNames.insert(e.GetName()).second;
            SLCR_VALIDATE(inserted, "Two or more executors have the same name: " << e.GetName());
        }

        // ActorSystem/LogConfig
        // TODO: implement log config validation
    }

    // ApiServer
    {
        SLCR_VALIDATE(config.Hasapi_server(), "api_server{} section cannot be empty");
        const auto& asConfig = config.api_server();

        // ApiServer/Port
        ui32 port = asConfig.GetPort(0);
        SLCR_VALIDATE(port > 1024, "Port " << port << " in api_server{} section is invalid");

        // ApiServer/ThreadPoolName
        const auto& threadPoolName = asConfig.GetThreadPoolName();
        SLCR_VALIDATE(!threadPoolName.Empty(), "ThreadPoolName in api_server{} section is not configured");
        SLCR_VALIDATE(ThreadPoolNames.contains(threadPoolName),
                      "In api_server{} section an unknown thread pool is used: " << threadPoolName);

        // ApiServer/MaxMessageSize
        SLCR_VALIDATE(asConfig.HasMaxMessageSize(), "MaxMessageSize in api_server{} section is not configured");
        ui64 maxMsgSize = FromProtoDataSize(asConfig.GetMaxMessageSize());
        SLCR_VALIDATE(maxMsgSize > 0, "MaxMessageSize in api_server{} section is not configured");
    }

    // TODO(ivanzhukov@): MonServer
}

} // namespace NSolomon::NSlicer
