#pragma once

#include <solomon/services/slicer/lib/common/reassignment_type.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>

namespace NSolomon::NSlicer::NDb {

struct TServiceConfigKey {
    TString Service;
    TString Cluster;
    TString Dc;

    TServiceConfigKey(TString service, TString cluster, TString dc) noexcept
        : Service{std::move(service)}
        , Cluster{std::move(cluster)}
        , Dc{std::move(dc)}
    {
    }

    bool operator==(const TServiceConfigKey& other) const = default;
};

struct TServiceConfig {
    TString Service;
    TString Cluster;
    TString Dc;
    bool IsFrozen;
    TDuration ReassignmentInterval;
    EReassignmentType ReassignmentType;
    ui32 MergeWhenMoreThanNumSlicesPerTask;
    double MergeKeyChurn;
    double MoveKeyChurn;
    double SplitSliceNTimesAsHotAsMean;
    ui32 SplitWhenFewerThanNumSlicesPerTask;

    TServiceConfig() = default;

    TServiceConfig(
            TString service,
            TString cluster,
            TString dc,
            bool isFrozen = false,
            // Default values are from the paper
            TDuration reassignmentInterval =  TDuration::Minutes(5),
            EReassignmentType reassignmentType = EReassignmentType::ByCpu,
            ui32 mergeWhenMoreThanNumSlicesPerTask = 50,
            double mergeKeyChurn = 0.01,
            double moveKeyChurn = 0.09,
            double splitSliceNTimesAsHotAsMean = 2,
            ui32 splitWhenFewerThanNumSlicesPerTask = 150) noexcept
        : Service{std::move(service)}
        , Cluster{std::move(cluster)}
        , Dc{std::move(dc)}
        , IsFrozen{isFrozen}
        , ReassignmentInterval{reassignmentInterval}
        , ReassignmentType{reassignmentType}
        , MergeWhenMoreThanNumSlicesPerTask{mergeWhenMoreThanNumSlicesPerTask}
        , MergeKeyChurn{mergeKeyChurn}
        , MoveKeyChurn{moveKeyChurn}
        , SplitSliceNTimesAsHotAsMean{splitSliceNTimesAsHotAsMean}
        , SplitWhenFewerThanNumSlicesPerTask{splitWhenFewerThanNumSlicesPerTask}
    {
    }

    TServiceConfigKey Key() const noexcept {
        return TServiceConfigKey{Service, Cluster, Dc};
    }

    bool operator==(const TServiceConfig& other) const = default;
};

} // namespace NSolomon::NSlicer::NDb

std::ostream& operator<<(std::ostream& os, const NSolomon::NSlicer::NDb::TServiceConfig& config);
IOutputStream& operator<<(IOutputStream& os, const NSolomon::NSlicer::NDb::TServiceConfig& config);
