#include <solomon/libs/cpp/ydb/util.h>
#include <solomon/services/slicer/lib/db/assignment_dao.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/stream/file.h>
#include <util/string/builder.h>

#include <algorithm>

namespace NSolomon::NSlicer::NDb {

using namespace NSlicer::NApi;

static NMonitoring::TMetricRegistry Registry;

class TDaoTest: public ::testing::Test {
protected:
    IAssignmentDaoPtr CreateDao(TStringBuf testName) {
        TString endpoint = TFileInput("ydb_endpoint.txt").ReadLine();
        TString database = TFileInput("ydb_database.txt").ReadLine();
        NYdb::TDriver driver(NYdb::TDriverConfig()
                                     .SetEndpoint(endpoint)
                                     .SetDatabase(database));

        TableClient_ = std::make_shared<NYdb::NTable::TTableClient>(driver);
        TablePath_ = TStringBuilder() << '/' << database << '/' << getpid() << "/Assignments/" << testName;

        Dao_ = CreateYdbAssignmentDao(TablePath_, TableClient_, Registry);
        Dao_->CreateTable().GetValueSync();

        return Dao_;
    }

    TVector<TAssignment> ReadSorted(const IAssignmentDao& dao) {
        auto ans = dao.LoadAll().ExtractValueSync();

        std::sort(ans.begin(), ans.end(), [](const auto& left, const auto& right) {
            return std::tie(left.Service, left.Cluster, left.Dc, left.Host)
                    < std::tie(right.Service, right.Cluster, right.Dc, right.Host);
        });

        return ans;
    }

protected:
    std::shared_ptr<NYdb::NTable::TTableClient> TableClient_;
    TString TablePath_;
    IAssignmentDaoPtr Dao_;
};

void UpdateVersion(TVector<TAssignment>& update, ui32 version) {
    for (auto& record: update) {
        record.Version = version;
    }
}

TEST_F(TDaoTest, Update) {
    auto dao = this->CreateDao("Update");

    auto ans = this->ReadSorted(*dao);
    auto ansBy = dao->Load("ingestor", "sts1", "Sas").ExtractValueSync();

    ASSERT_EQ(ans.size(), 0u);
    ASSERT_EQ(ans, ansBy);

    TVector<TAssignment> expected{
        {"ingestor", "sts1", "Sas", "ingestor-000", TSlices{ { 1, 10} }, 0},
        {"ingestor", "sts1", "Sas", "ingestor-001", TSlices{ {11, 20} }, 0},
        {"ingestor", "sts1", "Sas", "ingestor-002", TSlices{ {21, 30} }, 0},
        {"ingestor", "sts1", "Sas", "ingestor-003", TSlices{ {31, 40} }, 0},
    };

    dao->Update(expected).GetValueSync();

    ans = this->ReadSorted(*dao);
    ansBy = dao->Load("ingestor", "sts1", "Sas").ExtractValueSync();

    ASSERT_EQ(ans.size(), 4u);
    ASSERT_EQ(ans, ansBy);
    for (size_t i = 0; i < 4u; ++i) {
        SCOPED_TRACE(TStringBuilder() << "i = " << i << "; ");
        EXPECT_EQ(ans[i], expected[i]);
    }

    TVector<TAssignment> update{
        // add
        {"ingestor", "sts1", "Sas", "ingestor-000", TSlices{ {1, 10}, {41, 50} }, 1},
        // rm
        {"ingestor", "sts1", "Sas", "ingestor-001", TSlices{}, 1},
        // add
        {"ingestor", "sts1", "Sas", "ingestor-002", TSlices{ {11, 20} }, 1},
    };
    expected[0] = update[0];
    expected[1] = update[1];
    expected[2] = update[2];
    UpdateVersion(expected, update[0].Version);

    dao->Update(update).GetValueSync();

    ans = this->ReadSorted(*dao);
    ansBy = dao->Load("ingestor", "sts1", "Sas").ExtractValueSync();
    ASSERT_EQ(ans, ansBy);

    ASSERT_EQ(ans.size(), 4u);
    for (size_t i = 0; i < ans.size(); ++i) {
        SCOPED_TRACE(TStringBuilder() << "i = " << i << "; ");
        EXPECT_EQ(ans[i], expected[i]);
    }

    update = {
        {"ingestor", "sts1", "Sas", "ingestor-000", TSlices{ {41, 50} }, 2},
        {"ingestor", "sts1", "Sas", "ingestor-003", TSlices{}, 2},
    };
    expected[0] = update[0];
    expected[3] = update[1];
    UpdateVersion(expected, update[0].Version);

    dao->Update(update).GetValueSync();

    ans = this->ReadSorted(*dao);
    ansBy = dao->Load("ingestor", "sts1", "Sas").ExtractValueSync();
    ASSERT_EQ(ans, ansBy);

    ASSERT_EQ(ans.size(), 4u);
    for (size_t i = 0; i < ans.size(); ++i) {
        SCOPED_TRACE(TStringBuilder() << "i = " << i << "; ");
        EXPECT_EQ(ans[i], expected[i]);
    }

    update = {
        {"ingestor", "sts1", "Sas", "ingestor-004", TSlices{ {101, 110} }, 3},
    };
    expected.emplace_back(update[0]);
    UpdateVersion(expected, update[0].Version);

    dao->Update(update).GetValueSync();

    ans = this->ReadSorted(*dao);
    ansBy = dao->Load("ingestor", "sts1", "Sas").ExtractValueSync();
    ASSERT_EQ(ans, ansBy);

    ASSERT_EQ(ans.size(), 5u);
    for (size_t i = 0; i < ans.size(); ++i) {
        SCOPED_TRACE(TStringBuilder() << "i = " << i << "; ");
        EXPECT_EQ(ans[i], expected[i]);
    }

    // saving the same version one more time
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(dao->Update(update).GetValueSync(), yexception, "Trying to save stale data");

    UpdateVersion(update, 4u);

    TableClient_->RetryOperation<NYdb::NTable::TDataQueryResult>([path{TablePath_}] (NYdb::NTable::TSession s) mutable {
        // let's screw up the data
        TStringBuilder req;
        req << "UPDATE `" << path << '`' << R"(
            SET version = 5
            WHERE host = "ingestor-000";
        )";

        return NSolomon::NDb::ExecutePrepared(std::move(s), req, NYdb::TParamsBuilder{}.Build());
    }).GetValueSync();

    ASSERT_THROW_MESSAGE_HAS_SUBSTR(dao->Update(update).GetValueSync(), yexception, "Trying to save stale data");
}

} // namespace NSolomon::NSlicer::NDb
