#include <solomon/services/slicer/lib/db/ydb_service_config_dao.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/algorithm.h>
#include <util/stream/file.h>
#include <util/string/builder.h>

namespace NSolomon::NSlicer::NDb {

static NMonitoring::TMetricRegistry Registry;

namespace TMockYdb {
    static std::shared_ptr<IServiceConfigDao> Create(TStringBuf testName) {
        TString endpoint = TFileInput("ydb_endpoint.txt").ReadLine();
        TString database = TFileInput("ydb_database.txt").ReadLine();

        NYdb::TDriver driver(
                NYdb::TDriverConfig()
                        .SetEndpoint(endpoint)
                        .SetDatabase(database));

        auto tableClient = std::make_shared<NYdb::NTable::TTableClient>(driver);

        auto tablePath = TStringBuilder()
                << '/' << database << '/' << getpid() << "ServiceConfigs" << '/' << testName;

        auto dao = CreateYdbServiceConfigDao(std::move(tablePath), tableClient, Registry);
        dao->CreateTable().GetValueSync();
        return dao;
    }
};

class TServiceConfigDaoTest: public ::testing::Test {
protected:
    std::shared_ptr<IServiceConfigDao> CreateDao(TStringBuf testName) {
        return TMockYdb::Create(testName);
    }

    TVector<TServiceConfig> ReadSorted(const IServiceConfigDao& dao) {
        auto ans = dao.LoadAll().ExtractValueSync();
        Sort(ans, [](const auto& left, const auto& right) {
            return std::tie(left.Service, left.Cluster, left.Dc) < std::tie(right.Service, right.Cluster, right.Dc);
        });

        return ans;
    }
};

TEST_F(TServiceConfigDaoTest, Update) {
    auto dao = this->CreateDao("Update");

    auto ans = this->ReadSorted(*dao);
    auto ansBy1 = dao->Load({"ingestor", "sts1", "Iva"}).ExtractValueSync();
    auto ansBy2 = dao->Load({"ingestor", "sts2", "Iva"}).ExtractValueSync();

    ASSERT_EQ(ans.size(), 0u);
    ASSERT_FALSE(ansBy1);
    ASSERT_FALSE(ansBy2);

    TVector<TServiceConfig> init{
        {"ingestor", "sts1", "Iva", false},
        {"ingestor", "sts1", "Man", false},
        {"ingestor", "sts1", "Myt", false},
        {"ingestor", "sts1", "Sas", false},
        {"ingestor", "sts1", "Vla", false},

        {"ingestor", "sts2", "Iva", true},
        {"ingestor", "sts2", "Man", true},
        {"ingestor", "sts2", "Myt", true},
        {"ingestor", "sts2", "Sas", true},
        {"ingestor", "sts2", "Vla", true},
    };

    for (const auto& conf: init) {
        dao->Update(conf).GetValueSync();
    }

    ans = this->ReadSorted(*dao);

    ASSERT_EQ(ans.size(), init.size());
    for (size_t i = 0; i < init.size(); ++i) {
        EXPECT_EQ(ans[i], init[i]);
    }

    dao->Update(TServiceConfig{"ingestor", "sts1", "Iva", true}).GetValueSync();
    dao->Update(TServiceConfig{"ingestor", "sts2", "Iva", false}).GetValueSync();

    size_t sts1IvaIdx = 0;
    size_t sts2IvaIdx = init.size() / 2;

    ans = this->ReadSorted(*dao);
    ansBy1 = dao->Load(init[sts1IvaIdx].Key()).ExtractValueSync();
    ansBy2 = dao->Load(init[sts2IvaIdx].Key()).ExtractValueSync();

    ASSERT_EQ(ans.size(), init.size());
    for (size_t i = 0; i != init.size(); ++i) {
        if (i == sts1IvaIdx || i == sts2IvaIdx) {
            continue;
        }

        EXPECT_EQ(ans[i], init[i]);
    }

    ASSERT_TRUE(ansBy1);
    ASSERT_TRUE(ansBy2);
    EXPECT_EQ(init[sts1IvaIdx].Key(), ansBy1->Key());
    EXPECT_NE(init[sts1IvaIdx].IsFrozen, ansBy1->IsFrozen);
    EXPECT_EQ(init[sts2IvaIdx].Key(), ansBy2->Key());
    EXPECT_NE(init[sts2IvaIdx].IsFrozen, ansBy2->IsFrozen);

    dao->Update(init[sts1IvaIdx]).GetValueSync();
    dao->Update(init[sts2IvaIdx]).GetValueSync();

    ans = this->ReadSorted(*dao);
    ASSERT_EQ(ans.size(), init.size());
    for (size_t i = 0; i != init.size(); ++i) {
        EXPECT_EQ(ans[i], init[i]);
    }
}

} // namespace NSolomon::NSlicer::NDb
