#include "assignment_dao.h"

#include <solomon/libs/cpp/ydb/dao_base.h>
#include <solomon/libs/cpp/ydb/parser.h>
#include <solomon/libs/cpp/ydb/paged_reader.h>
#include <solomon/libs/cpp/ydb/util.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

using namespace NYdb;
using namespace NYdb::NTable;
using namespace NMonitoring;
using namespace NSolomon::NDb;

namespace NSolomon::NSlicer::NDb {
namespace {

using yandex::monitoring::slicer::HostSlices;

constexpr TStringBuf ASSIGNMENT_TABLE = "assignment_table";
constexpr TStringBuf ASSIGNMENT_UPDATE = "assignment_update";
constexpr TStringBuf ASSIGNMENT_LOAD = "assignment_load";

const TString COLUMN_SERVICE = "service";
const TString COLUMN_CLUSTER = "cluster";
const TString COLUMN_DC = "dc";
const TString COLUMN_HOST = "host";
const TString COLUMN_SLICES = "slices";
const TString COLUMN_VERSION = "version";

auto BytesToHostAssignmentParser(NYdb::TResultSetParser& parser, HostSlices (TAssignment::*field), const TString& column) {
    auto getter = [=, &parser] {
        return parser.ColumnParser(column).GetOptionalString(); // binary data
    };

    auto setter = [=] (TAssignment& model, const TString& bytes) {
        Y_VERIFY((model.*field).ParseFromString(bytes), "failed to parse a host's assignment from a DB");
    };

    return MakeHolder<TIndirectFieldParserImpl<TAssignment, decltype(setter), decltype(getter)>>(
            std::move(getter),
            std::move(setter));
}

class TAssignmentParser: public TModelParser<TAssignment> {
public:
    explicit TAssignmentParser(const TResultSet& rs)
        : TModelParser<TAssignment>{rs}
    {
        FieldMappings_[COLUMN_SERVICE] = Utf8FieldParser(Parser_, &TAssignment::Service, COLUMN_SERVICE);
        FieldMappings_[COLUMN_CLUSTER] = Utf8FieldParser(Parser_, &TAssignment::Cluster, COLUMN_CLUSTER);
        FieldMappings_[COLUMN_DC] = Utf8FieldParser(Parser_, &TAssignment::Dc, COLUMN_DC);
        FieldMappings_[COLUMN_HOST] = Utf8FieldParser(Parser_, &TAssignment::Host, COLUMN_HOST);
        FieldMappings_[COLUMN_SLICES] = BytesToHostAssignmentParser(Parser_, &TAssignment::HostSlices, COLUMN_SLICES);
        FieldMappings_[COLUMN_VERSION] = Uint32FieldParser(Parser_, &TAssignment::Version, COLUMN_VERSION);
    }
};

auto UnwrapReadTable = ParseReadTableResult<TAssignment, TAssignmentParser>;

class TYdbAssignmentDao: public TDaoBase, public IAssignmentDao {
public:
    TYdbAssignmentDao(TString tablePath, std::shared_ptr<TTableClient> client, TMetricRegistry& registry)
        : TDaoBase(std::move(client), registry)
        , TablePath_(std::move(tablePath))
    {
        Init();
    }

    TAsyncVoid CreateTable() override {
        const auto& query = RawQueries_.at(ASSIGNMENT_TABLE);
        return Execute<TAsyncStatus>(*Client_, [query] (TSession session) {
            return session.ExecuteSchemeQuery(query);
        }).Apply(CompleteStatus(CounterFor("createTable")));
    }

    TAsyncAssignments LoadAll() const override {
        return ReadTable<TAssignment>(
                *Client_,
                TablePath_,
                UnwrapReadTable,
                CounterFor("loadAll"));
    }

    // TODO(ivanzhukov): write a test checking that read data is correct even though the range didn't contain some columns
    // TODO(ivanzhukov): write a test that we didn't read data from the next service (i.e. after the specified service, cluster, dc)
    TAsyncAssignments Load(const TString& service, const TString& cluster, const TString& dc) override {
        TValue from = TValueBuilder()
                .BeginTuple()
                    .AddElement().OptionalUtf8(service)
                    .AddElement().OptionalUtf8(cluster)
                    .AddElement().OptionalUtf8(dc)
                .EndTuple()
                .Build();
        TValue to = TValueBuilder()
                .BeginTuple()
                    .AddElement().OptionalUtf8(service)
                    .AddElement().OptionalUtf8(cluster)
                    .AddElement().OptionalUtf8(dc)
                .EndTuple()
                .Build();

        NYdb::NTable::TReadTableSettings settings;
        settings
            .From(TKeyBound::Inclusive(from))
            .To(TKeyBound::Inclusive(to))
            .Ordered(true)
            ;

        auto reader = TAsyncPagedReader<TAssignment, decltype(UnwrapReadTable)>::Create(
                *Client_,
                TablePath_,
                UnwrapReadTable,
                CounterFor("load"));

        return reader->Read(settings);
    }

    // TODO(ivanzhukov): take service, cluster, dc, and version only once instead of copying it into every record
    TAsyncVoid Update(const TVector<TAssignment>& update) override {
        if (update.empty()) {
            return NThreading::MakeFuture();
        }

        TParamsBuilder pb;

        ui32 version = update[0].Version;
        TString service = update[0].Service;
        TString cluster = update[0].Cluster;
        TString dc = update[0].Dc;
        pb.AddParam("$version").Uint32(version).Build();
        pb.AddParam("$service").Utf8(service).Build();
        pb.AddParam("$cluster").Utf8(cluster).Build();
        pb.AddParam("$dc").Utf8(dc).Build();

        auto& addParam = pb.AddParam("$update").BeginList();
        TString slicesBuffer;

        for (const auto& upd: update) {
            Y_VERIFY(upd.Service == service, "multiple services within an update");
            Y_VERIFY(upd.Cluster == cluster, "multiple clusters within an update");
            Y_VERIFY(upd.Dc == dc, "multiple dcs within an update");
            Y_VERIFY(upd.Version == version, "multiple versions within an update");

            slicesBuffer.clear();
            Y_VERIFY(upd.HostSlices.SerializeToString(&slicesBuffer), "failed to serialize a host's assignment");

            addParam.AddListItem()
                    .BeginStruct()
                    .AddMember(COLUMN_SERVICE).Utf8(service)
                    .AddMember(COLUMN_CLUSTER).Utf8(cluster)
                    .AddMember(COLUMN_DC).Utf8(dc)
                    .AddMember(COLUMN_HOST).Utf8(upd.Host)
                    .AddMember(COLUMN_SLICES).String(slicesBuffer)
                    .AddMember(COLUMN_VERSION).Uint32(version)
                    .EndStruct();
        }

        addParam.EndList().Build();

        auto& req = RawQueries_.at(ASSIGNMENT_UPDATE);
        auto params = pb.Build();

        return Client_->RetryOperation<TDataQueryResult>([&req, params = std::move(params)] (NYdb::NTable::TSession s) mutable {
            return ExecutePrepared(std::move(s), req, params);
        }).Apply(CompleteStatus(CounterFor("update")));
    }

private:
    const TString& Name() const override {
        static const TString NAME{"AssignmentDao"};
        return NAME;
    }

    const TVector<TStringBuf>& Methods() const override {
        static const TVector<TStringBuf> METHODS{
                TStringBuf("createTable"),
                TStringBuf("loadAll"),
                TStringBuf("update"),
                TStringBuf("load")
        };
        return METHODS;
    }

    THashMap<TStringBuf, TString> LoadQueries() override {
        auto keys = {
            ASSIGNMENT_TABLE,
            ASSIGNMENT_UPDATE,
            ASSIGNMENT_LOAD
        };

        THashMap<TStringBuf, TString> result;

        for (const TStringBuf& key: keys) {
            TString query = NResource::Find(TString(key) + ".yql");
            Y_VERIFY_DEBUG(!query.empty(), "Resource %s.yql is empty", key.data());
            SubstGlobal(query, TStringBuf("${table.path}"), TablePath_);
            result.emplace(key, query);
        }

        return result;
    }

private:
    TString TablePath_;
};

} // namespace

IAssignmentDaoPtr CreateYdbAssignmentDao(
        TString tablePath,
        std::shared_ptr<TTableClient> client,
        TMetricRegistry& registry)
{
    return std::make_shared<TYdbAssignmentDao>(std::move(tablePath), std::move(client), registry);
}

} // namespace NSolomon::NSlicer::NDb
