#include "ydb_service_config_dao.h"

#include <solomon/libs/cpp/ydb/dao_base.h>
#include <solomon/libs/cpp/ydb/paged_reader.h>
#include <solomon/libs/cpp/ydb/parser.h>

#include <utility>

namespace NSolomon::NSlicer::NDb {
namespace {

using namespace NMonitoring;
using namespace NSolomon::NDb;
using namespace NYdb::NTable;
using namespace NYdb;

const TString NAME = "ServiceConfigDao";
constexpr TStringBuf METHOD_CREATE_TABLE = "createTable";
constexpr TStringBuf METHOD_LOAD_ALL = "loadAll";
constexpr TStringBuf METHOD_UPDATE = "update";
constexpr TStringBuf METHOD_LOAD = "load";
const TVector<TStringBuf> METHODS = {
        METHOD_CREATE_TABLE,
        METHOD_LOAD_ALL,
        METHOD_UPDATE,
        METHOD_LOAD,
};

constexpr TStringBuf SERVICE_CONFIG_CREATE_TABLE = "service_config_create_table";
constexpr TStringBuf SERVICE_CONFIG_LOAD = "service_config_load";
constexpr TStringBuf SERVICE_CONFIG_UPDATE = "service_config_update";

class TServiceConfigParser: public TModelParser<TServiceConfig> {
public:
    explicit TServiceConfigParser(const TResultSet& rs)
        : TModelParser<TServiceConfig>{rs}
    {
        const TString columnService = "service";
        const TString columnCluster = "cluster";
        const TString columnDc = "dc";
        const TString columnIsFrozen = "isFrozen";
        const TString columnReassignmentIntervalSeconds = "reassignmentIntervalSeconds";
        const TString columnReassignmentType = "reassignmentType";
        const TString columnMergeWhenMoreThanNumSlicesPerTask = "mergeWhenMoreThanNumSlicesPerTask";
        const TString columnMergeKeyChurn = "mergeKeyChurn";
        const TString columnMoveKeyChurn = "moveKeyChurn";
        const TString columnSplitSliceNTimesAsHotAsMean = "splitSliceNTimesAsHotAsMean";
        const TString columnSplitWhenFewerThanNumSlicesPerTask = "splitWhenFewerThanNumSlicesPerTask";

        FieldMappings_[columnService] = Utf8FieldParser(Parser_, &TServiceConfig::Service, columnService);
        FieldMappings_[columnCluster] = Utf8FieldParser(Parser_, &TServiceConfig::Cluster, columnCluster);
        FieldMappings_[columnDc] = Utf8FieldParser(Parser_, &TServiceConfig::Dc, columnDc);
        FieldMappings_[columnIsFrozen] = BoolFieldParser(Parser_, &TServiceConfig::IsFrozen, columnIsFrozen);
        FieldMappings_[columnReassignmentIntervalSeconds] = Uint32ToDurationFieldParser(
                Parser_,
                &TServiceConfig::ReassignmentInterval,
                columnReassignmentIntervalSeconds);
        FieldMappings_[columnReassignmentType] = Utf8ToEnumFieldParser(
                Parser_,
                &TServiceConfig::ReassignmentType,
                columnReassignmentType);
        FieldMappings_[columnMergeWhenMoreThanNumSlicesPerTask] = Uint32FieldParser(
                Parser_,
                &TServiceConfig::MergeWhenMoreThanNumSlicesPerTask,
                columnMergeWhenMoreThanNumSlicesPerTask);
        FieldMappings_[columnMergeKeyChurn] = DoubleFieldParser(
                Parser_,
                &TServiceConfig::MergeKeyChurn,
                columnMergeKeyChurn);
        FieldMappings_[columnMoveKeyChurn] = DoubleFieldParser(
                Parser_,
                &TServiceConfig::MoveKeyChurn,
                columnMoveKeyChurn);
        FieldMappings_[columnSplitSliceNTimesAsHotAsMean] = DoubleFieldParser(
                Parser_,
                &TServiceConfig::SplitSliceNTimesAsHotAsMean,
                columnSplitSliceNTimesAsHotAsMean);
        FieldMappings_[columnSplitWhenFewerThanNumSlicesPerTask] = Uint32FieldParser(
                Parser_,
                &TServiceConfig::SplitWhenFewerThanNumSlicesPerTask,
                columnSplitWhenFewerThanNumSlicesPerTask);
    }
};

auto UnwrapReadTable = ParseReadTableResult<TServiceConfig, TServiceConfigParser>;
auto UnwrapOne = ParseOne<TServiceConfig, TServiceConfigParser>;

class TServiceConfigDao: public TDaoBase, public IServiceConfigDao {
public:
    TServiceConfigDao(TString tablePath, std::shared_ptr<TTableClient> client, TMetricRegistry& registry)
        : TDaoBase(std::move(client), registry)
        , TablePath_(std::move(tablePath))
    {
        Init();
    }

    TAsyncVoid CreateTable() override {
        const auto& query = RawQueries_.at(SERVICE_CONFIG_CREATE_TABLE);

        return Execute<TAsyncStatus>(*Client_, [query] (TSession session) {
            return session.ExecuteSchemeQuery(query);
        }).Apply(CompleteStatus(CounterFor(METHOD_CREATE_TABLE)));
    }

    TAsyncServiceConfigs LoadAll() const override {
        return ReadTable<TServiceConfig>(
                *Client_,
                TablePath_,
                UnwrapReadTable,
                CounterFor(METHOD_LOAD_ALL));
    }

    TAsyncServiceConfig Load(const TServiceConfigKey& key) override {
        auto& req = RawQueries_.at(SERVICE_CONFIG_LOAD);
        auto pb = TParamsBuilder{}
                .AddParam("$service").Utf8(key.Service).Build()
                .AddParam("$dc").Utf8(key.Dc).Build()
                .AddParam("$cluster").Utf8(key.Cluster).Build()
                .Build();

        auto promise = NThreading::NewPromise<TAsyncServiceConfig::value_type>();
        ReadDataWithRetries(*Client_, req, std::move(pb), promise, UnwrapOne, CounterFor(METHOD_LOAD));

        return promise.GetFuture();
    }

    TAsyncVoid Update(const TServiceConfig& serviceConfig) override {
        auto params = NYdb::TParamsBuilder{}
                .AddParam("$service").Utf8(serviceConfig.Service).Build()
                .AddParam("$dc").Utf8(serviceConfig.Dc).Build()
                .AddParam("$cluster").Utf8(serviceConfig.Cluster).Build()
                .AddParam("$isFrozen").Bool(serviceConfig.IsFrozen).Build()
                .AddParam("$reassignmentIntervalSeconds").Uint32(serviceConfig.ReassignmentInterval.Seconds()).Build()
                .AddParam("$reassignmentType").Utf8(ToString(serviceConfig.ReassignmentType)).Build()
                .AddParam("$mergeWhenMoreThanNumSlicesPerTask").Uint32(serviceConfig.MergeWhenMoreThanNumSlicesPerTask).Build()
                .AddParam("$mergeKeyChurn").Double(serviceConfig.MergeKeyChurn).Build()
                .AddParam("$moveKeyChurn").Double(serviceConfig.MoveKeyChurn).Build()
                .AddParam("$splitSliceNTimesAsHotAsMean").Double(serviceConfig.SplitSliceNTimesAsHotAsMean).Build()
                .AddParam("$splitWhenFewerThanNumSlicesPerTask").Uint32(serviceConfig.SplitWhenFewerThanNumSlicesPerTask).Build()
                .Build();

        auto& req = RawQueries_.at(SERVICE_CONFIG_UPDATE);

        return Client_->RetryOperation<TDataQueryResult>([&req, params = std::move(params)] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), req, params);
        }).Apply(CompleteStatus(CounterFor(METHOD_UPDATE)));
    }

    // TODO(ivanzhukov): support the DELETE method when it'd be possible to delete a service from the UI

private:
    const TString& Name() const override {
        return NAME;
    }

    const TVector<TStringBuf>& Methods() const override {
        return METHODS;
    }

    THashMap<TStringBuf, TString> LoadQueries() override {
        auto keys = {
                SERVICE_CONFIG_CREATE_TABLE,
                SERVICE_CONFIG_LOAD,
                SERVICE_CONFIG_UPDATE,
        };

        THashMap<TStringBuf, TString> result;

        for (const TStringBuf& key: keys) {
            TString query = NResource::Find(TString(key) + ".yql");
            Y_VERIFY(!query.empty(), "Resource %s.yql is empty", key.data());
            SubstGlobal(query, TStringBuf("${table.path}"), TablePath_);
            result.emplace(key, query);
        }

        return result;
    }

private:
    const TString TablePath_;
};

} // namespace

std::shared_ptr<IServiceConfigDao> CreateYdbServiceConfigDao(
        TString tablePath,
        std::shared_ptr<NYdb::NTable::TTableClient> client,
        NMonitoring::TMetricRegistry& registry)
{
    return std::make_shared<TServiceConfigDao>(std::move(tablePath), std::move(client), registry);
}

}
