#include "host_status.h"

#include <solomon/services/slicer/lib/assignment_manager/events.h>
#include <solomon/services/slicer/lib/common/reassignment_type.h>

#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/string_utils/quote/quote.h>
#include <library/cpp/string_utils/url/url.h>

#include <util/string/split.h>
#include <util/system/hostname.h>

using namespace NActors;

namespace NSolomon::NSlicer {
namespace {

using namespace NSlicer::NApi;
using namespace yandex::monitoring::selfmon;

const TString HTTP_POST = "POST";
const TString ASSIGN_UNIFORMLY = "assignUniformly";
const TString REBALANCE_ONCE_BY_COUNT = "rebalanceByCount";
const TString REBALANCE_ONCE_BY_CPU = "rebalanceByCpu";
const TString REBALANCE_ONCE_BY_MEMORY = "rebalanceByMemory";
const TString SCHEDULE_REBALANCING = "scheduleRebalancing";
const TString REBALANCING_TYPE_CPU = "byCpu";
const TString REBALANCING_TYPE_COUNT = "byCount";
const TString REBALANCING_TYPE_MEMORY = "byMemory";

const absl::flat_hash_map<TString, EReassignmentType> Reassignments{
        {REBALANCING_TYPE_COUNT, EReassignmentType::ByCount},
        {REBALANCING_TYPE_CPU, EReassignmentType::ByCpu},
        {REBALANCING_TYPE_MEMORY, EReassignmentType::ByMemory}
};

const TString SETTINGS_IS_FROZEN = "settings.IsFrozen";
const TString SETTINGS_REASSIGNMENT_TYPE = "settings.ReassignmentType";
const TString SETTINGS_REASSIGNMENT_INTERVAL = "settings.ReassignmentInterval";
const TString SETTINGS_MERGE_SLICES_NUM = "settings.MergeWhenMoreThanNumSlicesPerTask";
const TString SETTINGS_MERGE_KEY_CHURN = "settings.MergeKeyChurn";
const TString SETTINGS_MOVE_KEY_CHURN = "settings.MoveKeyChurn";
const TString SETTINGS_SPLIT_TIMES_MEAN = "settings.SplitSliceNTimesAsHotAsMean";
const TString SETTINGS_SPLIT_SLICES_NUM = "settings.SplitWhenFewerThanNumSlicesPerTask";

const TString HELP_IS_FROZEN = "If not, rebalances load, according to the type, every time specified in the interval."
                               " If is, doesn't reassign anything, except when a reassign buton is pressed";
const TString HELP_REASSIGNMENT_TYPE = "By count — approximately the same number of keys will be assigned to every host";
const TString HELP_REASSIGNMENT_INTERVAL = "How often will Slicer rebalance a load for this service";
const TString HELP_MERGE_SLICES_NUM = "Within one round, merges adjacent cold slices as long as there are more "
                                      "than this num of slices per host _in aggregate_";
const TString HELP_MERGE_KEY_CHURN = "If such a fraction of keyspace is moved, stops a merging process";
const TString HELP_MOVE_KEY_CHURN = "If such a fraction of keyspace is moved, stops a rebalancing round";
const TString HELP_SPLIT_TIMES_MEAN = "If a slice is N times hotter than a mean slice, it'll be split";
const TString HELP_SPLIT_SLICES_NUM = "Within one round, splits hot slices as long as there are fewer than this num "
                                      "of slices per host _in aggregate_";

class THostStatusPageRequest: public NActors::TActor<THostStatusPageRequest> {
public:
    THostStatusPageRequest(TString service, NActors::TActorId serviceManager, TString hostName)
        : NActors::TActor<THostStatusPageRequest>(&TThis::StateFunc)
        , Service_{std::move(service)}
        , ServiceManager_(serviceManager)
        , HostName_(std::move(hostName))
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(TAssignmentEvents::THostInfoResult, OnResult);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        ReplyTo_ = ev->Sender;
        Send(ServiceManager_, new TAssignmentEvents::TGetHostInfo(std::move(Service_), HostName_));
    }

    void OnResult(TAssignmentEvents::THostInfoResult::TPtr ev) {
        Page page;
        page.set_title("Shards on " + HostName_);

        auto* table = page.mutable_component()->mutable_table();
        table->set_numbered(true);

        auto* shardColumn = table->add_columns();
        shardColumn->set_title("ShardId");
        auto* shardValues = shardColumn->mutable_int64();

        auto* cpuColumn = table->add_columns();
        cpuColumn->set_title("CPU");
        auto* cpuValues = cpuColumn->mutable_fractions();

        auto* memColumn = table->add_columns();
        memColumn->set_title("Memory");
        auto* memValues = memColumn->mutable_fractions();

        auto* netColumn = table->add_columns();
        netColumn->set_title("Network");
        auto* netValues = netColumn->mutable_data_size();

        for (const auto& node: ev->Get()->Shards) {
            shardValues->add_values(node.ShardId);

            auto* cpu = cpuValues->add_values();
            cpu->set_denom(node.CpuUsedOnHost);
            cpu->set_num(node.Cpu);
            cpu->set_view(Fraction_View_AsProgress);

            auto* mem = memValues->add_values();
            mem->set_denom(node.MemUsedOnHost);
            mem->set_num(node.Mem);
            mem->set_view(Fraction_View_AsProgress);

            netValues->add_values(node.Net);
        }

        Send(ReplyTo_, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        PassAway();
    }

private:
    TString Service_;
    NActors::TActorId ServiceManager_;
    NActors::TActorId ReplyTo_;
    TString HostName_;
};

class THostsPageListServicesRequest: public NActors::TActor<THostsPageListServicesRequest> {
public:
    explicit THostsPageListServicesRequest(NActors::TActorId serviceManager)
        : NActors::TActor<THostsPageListServicesRequest>(&TThis::StateFunc)
        , ServiceManager_(serviceManager)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(TServiceManagerEvents::TGetServicesResponse, OnResult);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        ReplyTo_ = ev->Sender;
        Send(ServiceManager_, new TServiceManagerEvents::TGetServices{});
    }

    void OnResult(TServiceManagerEvents::TGetServicesResponse::TPtr ev) {
        Page page;
        page.set_title("Hosts");

        const auto& services = ev->Get()->Services;
        if (services.empty()) {
            page.mutable_component()->mutable_value()->set_string("no services with hosts yet");
        } else {
            auto* refs = page.mutable_component()->mutable_list()->mutable_reference();

            for (const auto& service: services) {
                auto* val = refs->add_values();
                val->set_title(service);
                val->set_page("hosts");
                val->set_args("service=" + service);
            }
        }

        Send(ReplyTo_, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        PassAway();
    }

private:
    TString Service_;
    NActors::TActorId ServiceManager_;
    NActors::TActorId ReplyTo_;
};

class THostsPageRequest: public NActors::TActor<THostsPageRequest> {
public:
    explicit THostsPageRequest(TString port, TString service, TString activeTab, NActors::TActorId serviceManager)
        : NActors::TActor<THostsPageRequest>(&TThis::StateFunc)
        , Port_{port ? std::move(port) : "80"}
        , Service_{std::move(service)}
        , ActiveTab_{activeTab ? std::move(activeTab) : "hosts"}
        , ServiceManager_(serviceManager)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(TAssignmentEvents::TRedirectToLeader, OnRedirect);
            hFunc(TAssignmentEvents::TClusterStatusResult, OnResult);
        }
    }

private:
    static NDb::TServiceConfig ParseSettingsFromRequest(TStringBuf reqBody) {
        TVector<TStringBuf> tokens = StringSplitter(reqBody).SplitBySet("&=");
        NDb::TServiceConfig settings{};

        for (size_t i = 0; i <= tokens.size() - 2; i += 2) {
            auto& key = tokens[i];
            auto& value = tokens[i + 1];

            if (key == SETTINGS_IS_FROZEN) {
                settings.IsFrozen = true;
            } else if (key == SETTINGS_REASSIGNMENT_TYPE) {
                const auto it = Reassignments.find(TString{value});
                if (it == Reassignments.end()) {
                    ythrow yexception() << "unknown rebalancing type: " << value;
                }
                settings.ReassignmentType = it->second;
            } else if (key == SETTINGS_REASSIGNMENT_INTERVAL) {
                settings.ReassignmentInterval = TDuration::Seconds(FromString<ui64>(value));
            } else if (key == SETTINGS_MERGE_SLICES_NUM) {
                settings.MergeWhenMoreThanNumSlicesPerTask = FromString<ui32>(value);
            } else if (key == SETTINGS_MERGE_KEY_CHURN) {
                settings.MergeKeyChurn = FromString<double>(value);
            } else if (key == SETTINGS_MOVE_KEY_CHURN) {
                settings.MoveKeyChurn = FromString<double>(value);
            } else if (key == SETTINGS_SPLIT_TIMES_MEAN) {
                settings.SplitSliceNTimesAsHotAsMean = FromString<double>(value);
            } else if (key == SETTINGS_SPLIT_SLICES_NUM) {
                settings.SplitWhenFewerThanNumSlicesPerTask = FromString<ui32>(value);
            }
        }

        return settings;
    }

    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        ReplyTo_ = ev->Sender;
        auto req = ev->Get()->HttpReq;

        if (req->Method == HTTP_POST) {
            if (req->Body.starts_with(ASSIGN_UNIFORMLY)) {
                Send(ServiceManager_, new TAssignmentEvents::TAssignUniformly{Service_});
            } else if (req->Body.starts_with(REBALANCE_ONCE_BY_COUNT)) {
                Send(ServiceManager_, new TAssignmentEvents::TRebalanceOnceByCount{Service_});
            } else if (req->Body.starts_with(REBALANCE_ONCE_BY_CPU)) {
                Send(ServiceManager_, new TAssignmentEvents::TRebalanceOnceByCpu{Service_});
            } else if (req->Body.starts_with(REBALANCE_ONCE_BY_MEMORY)) {
                Send(ServiceManager_, new TAssignmentEvents::TRebalanceOnceByMemory{Service_});
            } else if (req->Body.starts_with(SCHEDULE_REBALANCING)) {
                Send(ServiceManager_, new TAssignmentEvents::TScheduleRebalancing{Service_});
            } else if (req->Body.starts_with("settings.")) {
                NDb::TServiceConfig settings = ParseSettingsFromRequest(req->Body);
                Send(ServiceManager_, new TAssignmentEvents::TChangeServiceSettings{Service_, std::move(settings)});
            }
        } else {
            TQuickCgiParam cgi{ev->Get()->HttpReq.Get()->Content};
            if (cgi.Has("freeze")) {
                // TODO: add an event to freeze the host
            }
            if (cgi.Has("active")) {
                // TODO: add an event to change the host state
            }
            if (cgi.Has("kick")) {
                // TODO: add an event to kick all shards from the host
            }
        }

        Send(ServiceManager_, new TAssignmentEvents::TGetClusterStatus{Service_});
    }

    static void AddInput(
            yandex::monitoring::selfmon::Form* form,
            InputType inputType,
            const TString& label,
            const TString& help,
            const TString& name,
            const TString& value)
    {
        auto* item = form->add_items();
        item->set_label(label);
        item->set_help(help);

        auto* input = item->mutable_input();
        input->set_type(inputType);
        input->set_name(name);
        input->set_value(value);
    }

    static void AddClusterStats(
            yandex::monitoring::selfmon::Grid_Row* row,
            const TAssignmentEvents::TClusterStatusResult& status)
    {
        auto* clusterInfo = row->add_columns()->mutable_component()->mutable_object();
        auto* clusterInfoFields = clusterInfo->mutable_fields();

        auto* leaderAddr = clusterInfoFields->Add();
        leaderAddr->set_name("Leader address");
        leaderAddr->mutable_value()->set_string(status.LeaderAddress);

        auto* lastReassignment = clusterInfoFields->Add();
        lastReassignment->set_name("Last reassignment was at");
        lastReassignment->mutable_value()->set_time(status.LastReassignment.GetValue());

        auto* lastReassignmentTook = clusterInfoFields->Add();
        lastReassignmentTook->set_name("Reassignment took");
        lastReassignmentTook->mutable_value()->set_duration(status.BalanceStats.Duration.GetValue());

        // merge
        auto* lastReassignmentMergeIterations = clusterInfoFields->Add();
        lastReassignmentMergeIterations->set_name("Merge iterations");
        lastReassignmentMergeIterations->mutable_value()->set_uint64(status.BalanceStats.MergeIterations);

        auto* lastReassignmentMergeKeyChurn = clusterInfoFields->Add();
        lastReassignmentMergeKeyChurn->set_name("Merge key churn");
        lastReassignmentMergeKeyChurn->mutable_value()->set_float64(status.BalanceStats.MergeKeyChurn);

        auto* lastReassignmentNumOfSlicesMerged = clusterInfoFields->Add();
        lastReassignmentNumOfSlicesMerged->set_name("Slices merged");
        lastReassignmentNumOfSlicesMerged->mutable_value()->set_uint64(status.BalanceStats.NumOfMergedSlices);

        auto* lastReassignmentMergeStatus = clusterInfoFields->Add();
        lastReassignmentMergeStatus->set_name("Merge status");
        lastReassignmentMergeStatus->mutable_value()->set_string(ToString(status.BalanceStats.MergeStatus));

        // move
        auto* lastReassignmentMoveIterations = clusterInfoFields->Add();
        lastReassignmentMoveIterations->set_name("Move iterations");
        lastReassignmentMoveIterations->mutable_value()->set_uint64(status.BalanceStats.MoveIterations);

        auto* lastReassignmentMoveKeyChurn = clusterInfoFields->Add();
        lastReassignmentMoveKeyChurn->set_name("Move key churn");
        lastReassignmentMoveKeyChurn->mutable_value()->set_float64(status.BalanceStats.MoveKeyChurn);

        auto* lastReassignmentNumOfSlicesMoved = clusterInfoFields->Add();
        lastReassignmentNumOfSlicesMoved->set_name("Slices moved");
        lastReassignmentNumOfSlicesMoved->mutable_value()->set_uint64(status.BalanceStats.NumOfMovedSlices);

        auto* lastReassignmentSplitIterations = clusterInfoFields->Add();
        lastReassignmentSplitIterations->set_name("Split iterations");
        lastReassignmentSplitIterations->mutable_value()->set_uint64(status.BalanceStats.SplitIterations);

        auto* lastReassignmentSplitCount = clusterInfoFields->Add();
        lastReassignmentSplitCount->set_name("Split count");
        lastReassignmentSplitCount->mutable_value()->set_uint64(status.BalanceStats.SplitCount);

        auto* lastReassignmentMeanLoad = clusterInfoFields->Add();
        lastReassignmentMeanLoad->set_name("Mean host load");
        TString meanLoad = FloatToString(status.BalanceStats.MeanLoadPercent, PREC_POINT_DIGITS, 2);
        lastReassignmentMeanLoad->mutable_value()->set_string(meanLoad + "%");

        auto* lastReassignmentLoadRSD = clusterInfoFields->Add();
        lastReassignmentLoadRSD->set_name("Host load RSD");
        TString loadRsd = FloatToString(status.BalanceStats.LoadRSDPercent, PREC_POINT_DIGITS, 2);
        lastReassignmentLoadRSD->mutable_value()->set_string(loadRsd + "%");

        if (auto& settings = status.ServiceSettings) {
            auto* keyChurnRatioBudgetMerge = clusterInfoFields->Add();
            keyChurnRatioBudgetMerge->set_name("Key churn ratio budget merge");
            keyChurnRatioBudgetMerge->mutable_value()->set_float64(settings->MergeKeyChurn);

            auto* keyChurnRatioBudgetMove = clusterInfoFields->Add();
            keyChurnRatioBudgetMove->set_name("Key churn ratio budget move");
            keyChurnRatioBudgetMove->mutable_value()->set_float64(settings->MoveKeyChurn);
        }
    }

    static void AddRebalancingTypeDropdown(
            yandex::monitoring::selfmon::Form* form,
            const NDb::TServiceConfig& settings)
    {
        auto* item = form->add_items();
        item->set_label("Rebalancing type");
        item->set_help(HELP_REASSIGNMENT_TYPE);
        auto* rebalancingType = item->mutable_select();
        rebalancingType->set_name(SETTINGS_REASSIGNMENT_TYPE);

        for (const auto& [name, type]: Reassignments) {
            auto* option = rebalancingType->add_options();
            option->set_title(name);
            option->set_value(name);
            option->set_selected(type == settings.ReassignmentType);
        }
    }

    static void AddSettingsEditor(yandex::monitoring::selfmon::Grid* grid, const NDb::TServiceConfig& settings) {
        auto* row = grid->add_rows();
        auto* col = row->add_columns();
        auto* form = col->mutable_component()->mutable_form();
        col->width();

        form->set_method(FormMethod::Post);

        auto* item = form->add_items();
        item->set_label("IsFrozen");
        item->set_help(HELP_IS_FROZEN);
        auto* isFrozen = item->mutable_check();
        isFrozen->set_name(SETTINGS_IS_FROZEN);
        isFrozen->set_value(settings.IsFrozen);

        AddRebalancingTypeDropdown(form, settings);
        AddInput(
                form,
                InputType::IntNumber,
                "Rebalancing interval (secs)",
                HELP_REASSIGNMENT_INTERVAL,
                SETTINGS_REASSIGNMENT_INTERVAL,
                ToString(settings.ReassignmentInterval.Seconds()));
        AddInput(
                form,
                InputType::IntNumber,
                "MergeWhenMoreThanNumSlicesPerTask",
                HELP_MERGE_SLICES_NUM,
                SETTINGS_MERGE_SLICES_NUM,
                ToString(settings.MergeWhenMoreThanNumSlicesPerTask));
        AddInput(
                form,
                InputType::FloatNumber,
                "MergeKeyChurn",
                HELP_MERGE_KEY_CHURN,
                SETTINGS_MERGE_KEY_CHURN,
                ToString(settings.MergeKeyChurn));
        AddInput(
                form,
                InputType::FloatNumber,
                "MoveKeyChurn",
                HELP_MOVE_KEY_CHURN,
                SETTINGS_MOVE_KEY_CHURN,
                ToString(settings.MoveKeyChurn));
        AddInput(
                form,
                InputType::FloatNumber,
                "SplitSliceNTimesAsHotAsMean",
                HELP_SPLIT_TIMES_MEAN,
                SETTINGS_SPLIT_TIMES_MEAN,
                ToString(settings.SplitSliceNTimesAsHotAsMean));
        AddInput(
                form,
                InputType::IntNumber,
                "SplitWhenFewerThanNumSlicesPerTask",
                HELP_SPLIT_SLICES_NUM,
                SETTINGS_SPLIT_SLICES_NUM,
                ToString(settings.SplitWhenFewerThanNumSlicesPerTask));

        auto* submit = form->add_submit();
        submit->set_title("Save settings");
        submit->set_name(""); // intentionally empty
        submit->set_value("saveSettings");
    }

    static void AddRebalancingControls(
            yandex::monitoring::selfmon::Grid* grid,
            const TAssignmentEvents::TClusterStatusResult& status)
    {
        auto& settings = status.ServiceSettings;

        if (!status.IsLeader || !settings) {
            return;
        }

        auto* firstRow = grid->add_rows();
        auto* rebalanceForm = firstRow->add_columns()->mutable_component()->mutable_form();
        rebalanceForm->set_method(FormMethod::Post);

        auto* assignUniformlyBtn = rebalanceForm->add_submit();
        assignUniformlyBtn->set_title("Assign uniformly");
        assignUniformlyBtn->set_name(ASSIGN_UNIFORMLY);
        assignUniformlyBtn->set_value("");

        auto* rebalanceByCountBtn = rebalanceForm->add_submit();
        rebalanceByCountBtn->set_title("Rebalance once by count");
        rebalanceByCountBtn->set_name(REBALANCE_ONCE_BY_COUNT);
        rebalanceByCountBtn->set_value("");

        auto* rebalanceByCpuBtn = rebalanceForm->add_submit();
        rebalanceByCpuBtn->set_title("Rebalance once by cpu");
        rebalanceByCpuBtn->set_name(REBALANCE_ONCE_BY_CPU);
        rebalanceByCpuBtn->set_value("");

        auto* rebalanceByMemoryBtn = rebalanceForm->add_submit();
        rebalanceByMemoryBtn->set_title("Rebalance once by memory");
        rebalanceByMemoryBtn->set_name(REBALANCE_ONCE_BY_MEMORY);
        rebalanceByMemoryBtn->set_value("");
    }

    void RenderTabs(Tabs* tabs) {
        if (auto* tab = tabs->add_tabs()) {
            tab->set_active(ActiveTab_ == "hosts");
            if (auto* ref = tab->mutable_reference()) {
                ref->set_title("Hosts");
                ref->set_page("/hosts");
                ref->set_args("tab=hosts&service=" + Service_);
            }
        }

        if (auto* tab = tabs->add_tabs()) {
            tab->set_active(ActiveTab_ == "config");
            if (auto* ref = tab->mutable_reference()) {
                ref->set_title("Config");
                ref->set_page("/hosts");
                ref->set_args("tab=config&service=" + Service_);
            }
        }
    }

    TString RequestUrl(TStringBuf fqdn) const {
        return TString::Join("http://"sv, fqdn, ':', Port_, "/selfmon/hosts?service="sv, Service_, "&tab="sv, ActiveTab_);
    }

    void OnRedirect(TAssignmentEvents::TRedirectToLeader::TPtr ev) {
        Send(ReplyTo_, new NSelfMon::TEvRedirectResp{RequestUrl(ev->Get()->LeaderFqdn)});
    }

    void OnResult(TAssignmentEvents::TClusterStatusResult::TPtr ev) {
        Page page;
        page.set_title("Hosts");

        if (auto* row = page.mutable_grid()->add_rows()) {
            auto* component = row->add_columns()->mutable_component();
            RenderTabs(component->mutable_tabs());
        }

        if (ActiveTab_ == "config") {
            auto* row = page.mutable_grid()->add_rows();
            if (ev->Get()->ServiceSettings) {
                AddSettingsEditor(row->add_columns()->mutable_grid(), *ev->Get()->ServiceSettings);
            }
            AddClusterStats(row->add_columns()->mutable_grid()->add_rows(), *ev->Get());

            Send(ReplyTo_, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
            PassAway();
            return;
        }

        if (auto* row = page.mutable_grid()->add_rows()) {
            AddRebalancingControls(row->add_columns()->mutable_grid(), *ev->Get());
        }

        auto* table = page.mutable_grid()->add_rows()->add_columns()->mutable_component()->mutable_table();
        table->set_numbered(true);

        auto* hostColumn = table->add_columns();
        hostColumn->set_title("Host");
        auto* hostValues = hostColumn->mutable_reference();

        auto* shardsColumn = table->add_columns();
        shardsColumn->set_title("Shards");
        auto* shardsValues = shardsColumn->mutable_uint64();

        auto* slicesColumn = table->add_columns();
        slicesColumn->set_title("Slices");
        auto* slicesValues = slicesColumn->mutable_uint64();

        auto* cpuColumn = table->add_columns();
        cpuColumn->set_title("CPU");
        auto* cpuValues = cpuColumn->mutable_fractions();

        auto* memColumn = table->add_columns();
        memColumn->set_title("Memory");
        auto* memValues = memColumn->mutable_fractions();

        auto* netColumn = table->add_columns();
        netColumn->set_title("Network");
        auto* netValues = netColumn->mutable_data_size();

        auto* stateColumn = table->add_columns();
        stateColumn->set_title("State");
        auto* stateValues = stateColumn->mutable_string();

        auto* buttonsColumn = table->add_columns();
        buttonsColumn->set_title("Actions");
        auto* buttonsValue = buttonsColumn->mutable_components();

        auto* lastSeenColumn = table->add_columns();
        lastSeenColumn->set_title("Last seen ago");
        auto* lastSeenValues = lastSeenColumn->mutable_duration();

        auto now = TActivationContext::Now();
        for (const auto& node: ev->Get()->Hosts) {
            auto* ref = hostValues->add_values();
            ref->set_title(node.Address);
            ref->set_page("/hosts");
            ref->set_args("host=" + node.Address + "&service=" + Service_);

            shardsValues->add_values(node.NumberOfShards);
            slicesValues->add_values(node.NumberOfSlices);

            auto* cpu = cpuValues->add_values();
            cpu->set_denom(node.CpuAll);
            cpu->set_num(node.Cpu);
            cpu->set_view(Fraction_View_AsProgress);

            auto* mem = memValues->add_values();
            mem->set_denom(node.MemAll);
            mem->set_num(node.Mem);
            mem->set_view(Fraction_View_AsProgress);

            netValues->add_values(node.Net);
            stateValues->add_values(ToString(node.NodeState));

            auto* Form = buttonsValue->add_values()->mutable_form();
            Form->set_method(FormMethod::Post);

            auto* freezeButton = Form->add_submit();
            freezeButton->set_name("freeze");
            freezeButton->set_title("Freeze");
            freezeButton->set_value(node.Address);

            auto* activeButton = Form->add_submit();
            activeButton->set_name("active");
            activeButton->set_title("Switch");
            activeButton->set_value(node.Address);

            auto* kickButton = Form->add_submit();
            kickButton->set_name("kick");
            kickButton->set_title("Kick");
            kickButton->set_value(node.Address);

            lastSeenValues->add_values((now - node.LastSeen).GetValue());
        }

        Send(ReplyTo_, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        PassAway();
    }

private:
    TString Port_;
    TString Service_;
    TString ActiveTab_;
    NActors::TActorId ServiceManager_;
    NActors::TActorId ReplyTo_;
};

class THostsPage: public NActors::TActor<THostsPage> {
public:
    explicit THostsPage(NActors::TActorId serviceManager)
        : NActors::TActor<THostsPage>(&TThis::StateFunc)
        , ServiceManager_(serviceManager)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(NActors::TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        TActorId reqId;

        if (auto service = ev->Get()->Param("service"); !service) {
            reqId = Register(new THostsPageListServicesRequest{ServiceManager_});
        } else {
            auto port = ev->Get()->HttpReq->Host.RAfter(':');
            auto host = ev->Get()->Param("host");

            if (host.empty()) {
                reqId = Register(new THostsPageRequest{
                        TString{port},
                        TString{service},
                        TString{ev->Get()->Param("tab")},
                        ServiceManager_});
            } else {
                reqId = Register(new THostStatusPageRequest{TString{service}, ServiceManager_, TString{host}});
            }
        }

        TActivationContext::Send(ev->Forward(reqId));
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    NActors::TActorId ServiceManager_;
};

class TFindShardHostPageRequest: public NActors::TActor<TFindShardHostPageRequest> {
public:
    TFindShardHostPageRequest(TString service, NActors::TActorId serviceManager, TString shardId)
        : NActors::TActor<TFindShardHostPageRequest>(&TThis::StateFunc)
        , Service_{std::move(service)}
        , ServiceManager_(serviceManager)
        , ShardId_(std::move(shardId))
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(TAssignmentEvents::TShardsHostInfoResult, OnResult);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        ReplyTo_ = ev->Sender;
        TNumId shardId = 0;
        TryFromString(ShardId_, shardId);
        Send(ServiceManager_, new TAssignmentEvents::TGetShardsHostInfo(std::move(Service_), shardId));
    }

    void OnResult(TAssignmentEvents::TShardsHostInfoResult::TPtr ev) {
        Page page;
        page.set_title("Search for the shard location");

        auto* grid = page.mutable_grid();

        auto* form = grid->add_rows()->add_columns()->mutable_component()->mutable_form();
        auto* shardIdInputItem = form->add_items();
        auto* shardIdInput = shardIdInputItem->mutable_input();
        shardIdInput->set_name("shardId");
        shardIdInput->set_value(ShardId_);
        shardIdInputItem->set_help("Enter only a numeric id of shard, e.g. 4033499062. If you don't have a numeric id, "
                                   "you can get it here https://solomon-pre.yandex-team.ru/admin/shards?state=RW");
        shardIdInputItem->set_label("Shard id");

        auto* button = form->add_submit();
        button->set_title("Find");

        if (!ShardId_.empty()) {
            auto* res = grid->add_rows()->add_columns()->mutable_component()->mutable_value();
            if (ev->Get()->HostName == "not found") {
                res->set_string("The search did not yield any results");
            } else {
                res->set_string("Shard " + ShardId_ + " is located on " + ev->Get()->HostName);
            }
        }

        Send(ReplyTo_, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        PassAway();
    }

private:
    TString Service_;
    NActors::TActorId ServiceManager_;
    NActors::TActorId ReplyTo_;
    TString ShardId_;
};

class TFindShardHostPage: public NActors::TActor<TFindShardHostPage> {
public:
    explicit TFindShardHostPage(NActors::TActorId serviceManager)
        : NActors::TActor<TFindShardHostPage>(&TThis::StateFunc)
        , ServiceManager_(serviceManager)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(NActors::TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        auto service = ev->Get()->Param("service");
        if (!service) {
            return; // TODO(ivanzhukov): list available services
        }

        auto shardId = ToString(ev->Get()->Param("shardId"));
        TActorId reqId = Register(new TFindShardHostPageRequest{TString{service}, ServiceManager_, shardId});

        TActivationContext::Send(ev->Forward(reqId));
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    NActors::TActorId ServiceManager_;
};

class TMoveShardPageRequest: public NActors::TActor<TMoveShardPageRequest> {
public:
    TMoveShardPageRequest(TString service, NActors::TActorId serviceManager)
        : NActors::TActor<TMoveShardPageRequest>(&TThis::StateFunc)
        , Service_{std::move(service)}
        , ServiceManager_(serviceManager)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(TAssignmentEvents::TValidateShardMoveParamsResult, OnResult);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        ShardId_ = ToString(ev->Get()->Param("shardId"));
        HostName_ = ToString(ev->Get()->Param("hostName"));
        UrlUnescape(HostName_);
        ReplyTo_ = ev->Sender;
        TNumId shardId = 0;
        TryFromString(ShardId_, shardId);
        Send(ServiceManager_, new TAssignmentEvents::TValidateShardMoveParams(Service_, shardId, HostName_));
    }

    void OnResult(TAssignmentEvents::TValidateShardMoveParamsResult::TPtr ev) {
        Page page;
        page.set_title("Move Shard");

        auto* grid = page.mutable_grid();

        auto* form = grid->add_rows()->add_columns()->mutable_component()->mutable_form();

        auto* shardIdInputItem = form->add_items();
        shardIdInputItem->mutable_input()->set_name("shardId");
        shardIdInputItem->set_label("Shard Id");
        shardIdInputItem->set_help("Enter only a numeric id of shard, e.g. 4033499062. If you don't have a numeric id, "
                                   "you can get it here https://solomon-pre.yandex-team.ru/admin/shards?state=RW");

        auto* hostNameInputItem = form->add_items();
        hostNameInputItem->mutable_input()->set_name("hostName");
        hostNameInputItem->set_label("Host name");
        hostNameInputItem->set_help("Use FQDN here with port number, e.g. 'ingestor-sts1-iva-020.mon.yandex.net:4760'");

        auto* button = form->add_submit();
        button->set_title("Move");

        if (ev->Get()->ShardIdValid && ev->Get()->HostValid) {
            Send(ServiceManager_, new TAssignmentEvents::TMoveShardRequest{
                Service_,
                FromString<TNumId>(ShardId_), HostName_});

            auto* res = grid->add_rows()->add_columns()->mutable_component()->mutable_code();
            res->set_content("The request was processed");
        } else {
            if (!ShardId_.empty() || !HostName_.empty()) {
                TString result;
                if (!HostName_.empty() && !ev->Get()->HostValid) {
                    result += "This host does not exist\n";
                }
                if (!ShardId_.empty() && !ev->Get()->ShardIdValid) {
                    result += "This shard does not exist\n";
                }
                if (ShardId_.empty() || HostName_.empty()) {
                    result = "Not all the data was entered";
                }
                if (!result.empty()) {
                    auto* res = grid->add_rows()->add_columns()->mutable_component()->mutable_code();
                    res->set_content(result);
                }
            }
        }

        Send(ReplyTo_, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        PassAway();
    }

private:
    TString Service_;
    NActors::TActorId ServiceManager_;
    NActors::TActorId ReplyTo_;
    TString ShardId_;
    TString HostName_;
};

class TMoveShardPage: public NActors::TActor<TMoveShardPage> {
public:
    explicit TMoveShardPage(NActors::TActorId serviceManager)
        : NActors::TActor<TMoveShardPage>(&TThis::StateFunc)
        , ServiceManager_(serviceManager)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(NActors::TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        auto service = ev->Get()->Param("service");
        if (!service) {
            return; // TODO(ivanzhukov): list available services
        }

        TActorId reqId = Register(new TMoveShardPageRequest{TString{service}, ServiceManager_});

        TActivationContext::Send(ev->Forward(reqId));
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    NActors::TActorId ServiceManager_;
};

} // namespace

std::unique_ptr<NActors::IActor> HostsPage(NActors::TActorId serviceManager) {
    return std::make_unique<THostsPage>(serviceManager);
}

std::unique_ptr<NActors::IActor> FindShardHostPage(NActors::TActorId serviceManager) {
    return std::make_unique<TFindShardHostPage>(serviceManager);
}

std::unique_ptr<NActors::IActor> MoveShardPage(NActors::TActorId serviceManager) {
    return std::make_unique<TMoveShardPage>(serviceManager);
}


} // namespace NSolomon::NSlicer
