package ru.yandex.stockpile.api.grpc;

import java.util.List;

import com.google.common.collect.ImmutableList;
import io.grpc.ServerServiceDefinition;

import ru.yandex.concurrency.limits.actors.Limiter;
import ru.yandex.grpc.utils.ExternalGrpcService;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.grpc.handler.AllocateLocalIdsHandler;
import ru.yandex.stockpile.api.grpc.handler.BulkShardCommandHandler;
import ru.yandex.stockpile.api.grpc.handler.DeleteMetricByShardHandler;
import ru.yandex.stockpile.api.grpc.handler.DeleteMetricDataHandler;
import ru.yandex.stockpile.api.grpc.handler.DeleteMetricHandler;
import ru.yandex.stockpile.api.grpc.handler.Handler;
import ru.yandex.stockpile.api.grpc.handler.ReadCompressedManyHandler;
import ru.yandex.stockpile.api.grpc.handler.ReadCompressedOneHandler;
import ru.yandex.stockpile.api.grpc.handler.ReadMetricsMetaHandler;
import ru.yandex.stockpile.api.grpc.handler.ReadUncompressedManyHandler;
import ru.yandex.stockpile.api.grpc.handler.ReadUncompressedOneHandler;
import ru.yandex.stockpile.api.grpc.handler.ServerStatusHandler;
import ru.yandex.stockpile.api.grpc.handler.WriteCompressedOneHandler;
import ru.yandex.stockpile.api.grpc.handler.WriteDataBinaryHandler;
import ru.yandex.stockpile.api.grpc.handler.WriteLogHandler;
import ru.yandex.stockpile.api.grpc.handler.WriteOneHandler;
import ru.yandex.stockpile.api.read.StockpileReadApi;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;

import static io.grpc.stub.ServerCalls.asyncUnaryCall;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcStockpileServiceFactory {

    public static ExternalGrpcService create(StockpileReadApi readApi, Limiter readLimiter, StockpileLocalShards shards, MetricRegistry registry) {
        ServerServiceDefinition.Builder defBuilder = ServerServiceDefinition.builder(StockpileServiceGrpc.SERVICE_NAME);
        for (Handler handler : createHandlers(readApi, readLimiter, shards)) {
            defBuilder.addMethod(handler.descriptor(), asyncUnaryCall(new UnaryServerCallHandler<>(handler, registry)));
        }

        ServerServiceDefinition def = defBuilder.build();
        return () -> def;
    }

    private static List<Handler> createHandlers(StockpileReadApi readApi, Limiter readLimiter, StockpileLocalShards shards) {
        return ImmutableList.<Handler>builder()
            .add(new BulkShardCommandHandler(shards))
            .add(new DeleteMetricDataHandler(shards))
            .add(new DeleteMetricHandler(shards))
            .add(new ReadUncompressedOneHandler(shards, readApi, readLimiter))
            .add(new ReadCompressedOneHandler(shards, readApi, readLimiter))
            .add(new ReadUncompressedManyHandler(shards, readApi, readLimiter))
            .add(new ReadCompressedManyHandler(shards, readApi, readLimiter))
            .add(new ServerStatusHandler(shards))
            .add(new WriteCompressedOneHandler(shards))
            .add(new WriteOneHandler(shards))
            .add(new ReadMetricsMetaHandler(shards))
            .add(new WriteDataBinaryHandler(shards))
            .add(new AllocateLocalIdsHandler(shards))
            .add(new WriteLogHandler(shards))
            .add(new DeleteMetricByShardHandler(shards))
            .build();
    }
}
