package ru.yandex.stockpile.api.grpc;

import java.io.IOException;

import com.google.protobuf.CodedInputStream;

import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.stockpile.api.EStockpileStatusCode;

/**
 * @author Vladimir Gordiychuk
 */
public class HistogramParser {
    private final static Rate nonUniqueBounds = MetricRegistry.root().rate("stockpile.histogram.nonUniqueBound");

    public static Histogram parse(CodedInputStream input) {
        return parse(input, Histogram.newInstance());
    }

    public static Histogram parse(CodedInputStream input, Histogram histogram) {
        try {
            boolean hasNonUniqueBounds = false;
            double prevBound = -Double.MAX_VALUE;
            int boundsIndex = 0;
            int bucketIndex = 0;
            boolean done = false;
            while (!done) {
                int tag = input.readTag();
                switch (tag) {
                    case 0:
                        done = true;
                        break;
                    case 9: {
                        double bound = input.readDouble();
                        histogram.setUpperBound(boundsIndex++, bound);
                        if (prevBound > bound) {
                            throw new StockpileRuntimeException(EStockpileStatusCode.INVALID_REQUEST, "bounds unsorted: " + bound + " prev " + prevBound, false);
                        } else if (prevBound == bound) {
                            hasNonUniqueBounds = true;
                        }
                        prevBound = bound;
                        break;
                    }
                    case 10: {
                        int length = input.readRawVarint32();
                        int limit = input.pushLimit(length);
                        while (input.getBytesUntilLimit() > 0) {
                            double bound = input.readDouble();
                            histogram.setUpperBound(boundsIndex++, bound);
                            if (prevBound > bound) {
                                throw new StockpileRuntimeException(EStockpileStatusCode.INVALID_REQUEST, "bounds unsorted: " + bound + " prev " + prevBound, false);
                            } else if (prevBound == bound) {
                                hasNonUniqueBounds = true;
                            }
                            prevBound = bound;
                        }
                        input.popLimit(limit);
                        break;
                    }
                    case 16: {
                        histogram.setBucketValue(bucketIndex++, input.readUInt64());
                        break;
                    }
                    case 18: {
                        int length = input.readRawVarint32();
                        int limit = input.pushLimit(length);
                        while (input.getBytesUntilLimit() > 0) {
                            histogram.setBucketValue(bucketIndex++, input.readUInt64());
                        }
                        input.popLimit(limit);
                        break;
                    }
                    default: {
                        if (!input.skipField(tag)) {
                            done = true;
                        }
                        break;
                    }
                }
            }

            if (hasNonUniqueBounds) {
                nonUniqueBounds.inc();
            }

            return histogram;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
