package ru.yandex.stockpile.api.grpc;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillClose;

import io.grpc.Server;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.grpc.utils.ExternalGrpcService;
import ru.yandex.grpc.utils.GrpcServerFactory;
import ru.yandex.grpc.utils.InternalGrpcService;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcServerConfig;
import ru.yandex.solomon.config.protobuf.rpc.TRpcServerConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.grpc.GrpcContext;
import ru.yandex.stockpile.cluster.balancer.StockpileBalancerContext;
import ru.yandex.stockpile.tool.StockpileToolContext;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@Import({
    StockpileGrpcServiceContext.class,
    StockpileBalancerContext.class,
    StockpileToolContext.class,
    GrpcContext.class
})
@ParametersAreNonnullByDefault
public class StockpileGrpcServerContext implements DisposableBean {
    private final TRpcServerConfig config;
    private final GrpcServerFactory serverFactory;
    private final ThreadPoolProvider threads;

    @WillClose
    private Server externalServer;
    @WillClose
    private Server internalServer;

    public StockpileGrpcServerContext(TRpcServerConfig config, GrpcServerFactory serverFactory, ThreadPoolProvider threads) {
        this.config = config;
        this.serverFactory = serverFactory;
        this.threads = threads;
    }

    @Bean
    public Server grpcExternalServer(List<ExternalGrpcService> services) {
        String prefix = "RpcServerConfig.GrpcServerConfig";
        TGrpcServerConfig config = this.config.getGrpcServerConfig();
        externalServer = serverFactory.makeServer(prefix, config, services);
        serverFactory.startWithTimeout(config, externalServer);
        return externalServer;
    }

    @Bean
    public Server grpcInternalServer(@Autowired(required = false) List<InternalGrpcService> services) {
        String prefix = "RpcServerConfig.InternalGrpcServerConfig";
        TGrpcServerConfig config = this.config.getInternalGrpcServerConfig();
        internalServer = serverFactory.makeServer(prefix, config, services);
        serverFactory.startWithTimeout(config, internalServer);
        return internalServer;
    }

    @Override
    public void destroy() {
        if (externalServer != null) {
            externalServer.shutdown();
        }

        if (internalServer != null) {
            internalServer.shutdown();
        }
    }
}
