package ru.yandex.stockpile.api.grpc;

import java.time.Duration;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.concurrency.limits.actors.Limiter;
import ru.yandex.concurrency.limits.actors.LimiterImpl;
import ru.yandex.grpc.utils.ExternalGrpcService;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.stockpile.TReadConfig;
import ru.yandex.stockpile.api.read.StockpileReadApi;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@Import({
    StockpileLocalShards.class,
    StockpileReadApi.class
})
@ParametersAreNonnullByDefault
public class StockpileGrpcServiceContext {

    private final StockpileReadApi readApi;
    private final StockpileLocalShards shards;
    private final MetricRegistry registry;

    public StockpileGrpcServiceContext(StockpileReadApi readApi, StockpileLocalShards shards, MetricRegistry registry) {
        this.readApi = readApi;
        this.shards = shards;
        this.registry = registry;
    }

    @Bean
    public ExternalGrpcService stockpileService(Limiter readLimiter) {
        return GrpcStockpileServiceFactory.create(readApi, readLimiter, shards, registry);
    }

    @Bean
    Limiter readLimiter(Optional<TReadConfig> config) {
        int minLimit = config.map(TReadConfig::getGrpcReadRequestMinInFlight).filter(v -> v > 0).orElse(20);
        int maxLimit = config.map(TReadConfig::getGrpcReadRequestInFlight).filter(v -> v > 0).orElse(10000);
        double tolerance = config.map(TReadConfig::getGrpcReadRttTolerance).filter(v -> v > 0).orElse(5.0);
        var minRtt = config.map(TReadConfig::getGrpcMinRtt).map(TimeConverter::protoToDuration).filter(d -> !d.isZero()).orElse(Duration.ofMillis(100));
        return LimiterImpl.newBuilder()
                .maxLimit(maxLimit)
                .minLimit(minLimit)
                .initLimit(minLimit)
                .minRtt(minRtt.toNanos())
                .rttTolerance(tolerance)
                .registry(registry)
                .operation("api_read")
                .build();
    }

}
