package ru.yandex.stockpile.api.grpc;

import java.io.IOException;

import com.google.protobuf.CodedInputStream;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.WireFormat;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.model.type.SummaryDouble;
import ru.yandex.stockpile.api.TPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class TPointParser {

    public static AggrPoint parse(CodedInputStream input) {
        AggrPoint point = new AggrPoint();
        parse(point, input);
        return point;
    }

    public static void parse(AggrPoint point, CodedInputStream input) {
        try {
            point.resetToDefaultPrimitives();
            point.columnSet = 0;
            do {
                int tag = input.readTag();
                if (tag == 0) {
                    return;
                }

                switch (WireFormat.getTagFieldNumber(tag)) {
                    case TPoint.TIMESTAMPSMILLIS_FIELD_NUMBER:
                        point.setTsMillis(input.readUInt64());
                        break;
                    case TPoint.DOUBLEVALUE_FIELD_NUMBER:
                        point.setValue(input.readDouble());
                        break;
                    case TPoint.LOGHISTOGRAM_FIELD_NUMBER: {
                        int limit = beforeMessage(input);
                        point.setLogHistogram(LogHistogramParser.parse(input, LogHistogram.orNew(point.logHistogram)));
                        afterMessage(input, limit);
                        break;
                    }
                    case TPoint.STEPMILLIS_FIELD_NUMBER:
                        point.setStepMillis(input.readUInt64());
                        break;
                    case TPoint.MERGE_FIELD_NUMBER:
                        point.setMerge(input.readBool());
                        break;
                    case TPoint.COUNT_FIELD_NUMBER:
                        point.setCount(input.readUInt32());
                        break;
                    case TPoint.HISTOGRAM_FIELD_NUMBER: {
                        int limit = beforeMessage(input);
                        point.setHistogram(HistogramParser.parse(input, Histogram.orNew(point.histogram)));
                        afterMessage(input, limit);
                        break;
                    }
                    case TPoint.SUMMARYINT64_FIELD_NUMBER: {
                        int limit = beforeMessage(input);
                        point.setSummaryInt64(SummaryInt64Parser.parse(input));
                        afterMessage(input, limit);
                        break;
                    }
                    case TPoint.SUMMARYDOUBLE_FIELD_NUMBER: {
                        int limit = beforeMessage(input);
                        point.setSummaryDouble(SummaryDoubleParser.parse(input, SummaryDouble.orNew(point.summaryDouble)));
                        afterMessage(input, limit);
                        break;
                    }
                    case TPoint.LONGVALUE_FIELD_NUMBER:
                        point.setLongValue(input.readUInt64());
                        break;
                    default:
                        if (!input.skipField(tag)) {
                            return;
                        }
                        break;
                }
            } while (true);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static int beforeMessage(CodedInputStream input) throws IOException {
        int length = input.readRawVarint32();
        return input.pushLimit(length);
    }

    private static void afterMessage(CodedInputStream input, int prevLimit) throws InvalidProtocolBufferException {
        input.checkLastTagWas(0);
        input.popLimit(prevLimit);
    }
}
