package ru.yandex.stockpile.api.grpc;

import java.io.IOException;

import com.google.protobuf.CodedInputStream;
import com.google.protobuf.WireFormat;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.grpc.handler.LogEntryAppender;
import ru.yandex.stockpile.server.shard.StockpileWriteRequest;

/**
 * @author Vladimir Gordiychuk
 */
public class TWriteRequestParser {

    public static int parse(StockpileWriteRequest.Builder builder, AggrPoint point, CodedInputStream input) {
        long localId = 0;
        MetricArchiveMutable archive = new MetricArchiveMutable();
        point.reset();

        try {
            boolean done = false;
            while (!done) {
                int tag = input.readTag();
                if (tag == 0) {
                    break;
                }

                switch (WireFormat.getTagFieldNumber(tag)) {
                    case TWriteRequest.METRIC_ID_FIELD_NUMBER: {
                        int length = input.readRawVarint32();
                        final int oldLimit = input.pushLimit(length);
                        localId = MetricIdParser.parseLocalId(input);
                        input.checkLastTagWas(0);
                        input.popLimit(oldLimit);
                        break;
                    }
                    case TWriteRequest.COLUMNMASK_FIELD_NUMBER:
                        archive.ensureCapacity(input.readUInt32(), 1);
                        break;
                    case TWriteRequest.OWNERSHARDID_FIELD_NUMBER:
                        archive.setOwnerShardId(input.readUInt32());
                        break;
                    case TWriteRequest.TYPE_FIELD_NUMBER:
                        MetricType type = MetricType.forNumber(input.readEnum());
                        if (archive.getType() != type) {
                            MetricArchiveMutable copy = new MetricArchiveMutable();
                            copy.setType(type);
                            copy.addAll(archive);
                            archive = copy;
                        }
                        break;
                    case TWriteRequest.POINTSOLD_FIELD_NUMBER:
                    case TWriteRequest.POINTS_FIELD_NUMBER: {
                        int length = input.readRawVarint32();
                        final int oldLimit = input.pushLimit(length);

                        TPointParser.parse(point, input);
                        archive.addRecord(point);

                        input.checkLastTagWas(0);
                        input.popLimit(oldLimit);
                        break;
                    }
                    default:
                        if (!input.skipField(tag)) {
                            done = true;
                        }
                        break;
                }
            }

            LogEntryAppender.fillHeaders(archive);
            builder.addArchive(localId, archive);
            return archive.getRecordCount();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
