package ru.yandex.stockpile.api.grpc;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import io.grpc.stub.ServerCalls;
import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.grpc.handler.Handler;

import static ru.yandex.stockpile.api.grpc.handler.Handlers.classifyError;

/**
 * @author Vladimir Gordiychuk
 */
public class UnaryServerCallHandler<ReqT, RespT> implements ServerCalls.UnaryMethod<ReqT, RespT> {
    private final Logger logger = LoggerFactory.getLogger(UnaryServerCallHandler.class);
    private final ConcurrentMap<EStockpileStatusCode, Rate> counterByStatus = new ConcurrentHashMap<>();
    private final Handler<ReqT, RespT> handler;
    private final MetricRegistry registry;

    public UnaryServerCallHandler(Handler<ReqT, RespT> handler, MetricRegistry registry) {
        this.handler = handler;
        this.registry = registry;
    }

    @Override
    public void invoke(ReqT request, StreamObserver<RespT> observer) {
        try {
            handler.unaryCall(request)
                .whenComplete((response, throwable) -> {
                    final EStockpileStatusCode statusCode;
                    if (throwable != null) {
                        observer.onError(throwable);
                        statusCode = classifyError(throwable);
                    } else {
                        observer.onNext(response);
                        observer.onCompleted();
                        statusCode = handler.getStatusCode(response);
                    }

                    incError(statusCode);
                });
        } catch (Throwable e) {
            var status = classifyError(e);
            incError(status);
            logger.info("Call {} failed with status {}", handler, status, e);
            observer.onError(e);
        }
    }

    private void incError(EStockpileStatusCode statusCode) {
        counterByStatus.computeIfAbsent(statusCode, code -> registry.subRegistry("endpoint", handler.descriptor().getFullMethodName())
            .subRegistry("code", code.name())
            .rate("stockpileServer.responseStatus"))
            .inc();
    }
}
