package ru.yandex.stockpile.api.grpc.handler;

import java.util.concurrent.CompletableFuture;

import io.grpc.MethodDescriptor;
import it.unimi.dsi.fastutil.longs.LongArrayList;

import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TAllocateLocalIdsRequest;
import ru.yandex.stockpile.api.TAllocateLocalIdsResponse;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

/**
 * @author Vladimir Gordiychuk
 */
public class AllocateLocalIdsHandler extends ShardRequestHandler<TAllocateLocalIdsRequest, TAllocateLocalIdsResponse> {

    public AllocateLocalIdsHandler(StockpileLocalShards shards) {
        super(shards);
    }

    @Override
    protected CompletableFuture<TAllocateLocalIdsResponse> unaryCall(StockpileShard shard, TAllocateLocalIdsRequest request) {
        var state = shard.getLoadState();
        if (state != StockpileShard.LoadState.DONE) {
            return CompletableFuture.completedFuture(response(EStockpileStatusCode.SHARD_NOT_READY, state.name()));
        }

        return shard.allocateLocalIds(request.getSize(), request.getDeadline())
            .thenApply(localIds -> TAllocateLocalIdsResponse.newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .setList(TAllocateLocalIdsResponse.TList.newBuilder()
                    .addAllLocalIds(LongArrayList.wrap(localIds))
                    .build())
                .build());
    }

    @Override
    protected int shardId(TAllocateLocalIdsRequest request) {
        return request.getShardId();
    }

    @Override
    protected TAllocateLocalIdsResponse response(EStockpileStatusCode status, String details) {
        return TAllocateLocalIdsResponse.newBuilder()
                .setStatus(status)
                .setStatusMessage(details)
                .build();
    }

    @Override
    public MethodDescriptor<TAllocateLocalIdsRequest, TAllocateLocalIdsResponse> descriptor() {
        return StockpileServiceGrpc.getAllocateLocalIdsMethod();
    }

    @Override
    public EStockpileStatusCode getStatusCode(TAllocateLocalIdsResponse response) {
        return response.getStatus();
    }
}
