package ru.yandex.stockpile.api.grpc.handler;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;

import io.grpc.MethodDescriptor;
import io.grpc.protobuf.ProtoUtils;

import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TShardCommandResponse;
import ru.yandex.stockpile.api.grpc.mem.ShardBatchRequest;
import ru.yandex.stockpile.api.grpc.mem.ShardBatchRequestMarshaller;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

/**
 * @author Vladimir Gordiychuk
 */
public class BulkShardCommandHandler extends ShardRequestHandler<ShardBatchRequest, TShardCommandResponse> {
    private static final io.grpc.MethodDescriptor<ShardBatchRequest, TShardCommandResponse> BINARY_METHOD_BULK_SHARD_COMMAND =
            MethodDescriptor.<ShardBatchRequest, TShardCommandResponse>newBuilder()
                    .setType(MethodDescriptor.MethodType.UNARY)
                    .setFullMethodName(StockpileServiceGrpc.getBulkShardCommandMethod().getFullMethodName())
                    .setRequestMarshaller(new ShardBatchRequestMarshaller())
                    .setResponseMarshaller(ProtoUtils.marshaller(TShardCommandResponse.getDefaultInstance()))
                    .build();

    public BulkShardCommandHandler(StockpileLocalShards shards) {
        super(shards);
    }

    @Override
    protected CompletableFuture<TShardCommandResponse> unaryCall(StockpileShard shard, ShardBatchRequest request) {
        shard.metrics.utimeNanos.mark(request.parseTimeNanos);
        if (!shard.canServeWrites()) {
            return CompletableFuture.completedFuture(response(EStockpileStatusCode.SHARD_NOT_READY, shard.getLoadState().name()));
        }

        if (request.isEmpty()) {
            return CompletableFuture.completedFuture(response(EStockpileStatusCode.INVALID_REQUEST, "empty write request"));
        }

        var req = Objects.requireNonNull(request.getContent());
        return shard.pushBatch(req)
            .thenApply(result -> response(EStockpileStatusCode.OK));
    }

    @Override
    protected int shardId(ShardBatchRequest request) {
        return request.getShardId();
    }

    @Override
    protected TShardCommandResponse response(EStockpileStatusCode status, String details) {
        return TShardCommandResponse.newBuilder()
                .setStatus(status)
                .setStatusMessage(details)
                .build();
    }

    @Override
    public MethodDescriptor<ShardBatchRequest, TShardCommandResponse> descriptor() {
        return BINARY_METHOD_BULK_SHARD_COMMAND;
    }

    @Override
    public EStockpileStatusCode getStatusCode(TShardCommandResponse response) {
        return response.getStatus();
    }
}
