package ru.yandex.stockpile.api.grpc.handler;

import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.kikimr.client.kv.KikimrKvGenerationChangedRuntimeException;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;
import ru.yandex.solomon.codec.CorruptedBinaryDataRuntimeException;
import ru.yandex.solomon.codec.UnsupportedFormatRuntimeException;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;

/**
 * @author Vladimir Gordiychuk
 */
public class Handlers {
    private static final Logger logger = LoggerFactory.getLogger(Handlers.class);

    public static <ReqT extends Message, RespT> void logError(Handler<ReqT, RespT> handler, ReqT request, Throwable error) {
        logError(handler, TextFormat.shortDebugString(request), error);
    }

    public static <ReqT, RespT> void logError(Handler<ReqT, RespT> handler, String request, Throwable error) {
        EStockpileStatusCode code = classifyError(error);
        String endpoint = handler.descriptor().getFullMethodName();
        if (isWarn(code)) {
            logger.warn("{} - {} for request {}", code, endpoint, request);
        } else {
            logger.error("{} - {} for request {}", code, endpoint, request, error);
        }
    }

    private static boolean isWarn(EStockpileStatusCode code) {
        switch (code) {
            case DEADLINE_EXCEEDED:
            case SHARD_NOT_READY:
            case SHARD_ABSENT_ON_HOST:
                return true;

            default:
                return false;
        }
    }

    public static EStockpileStatusCode classifyError(Throwable throwable) {
        Throwable cause = CompletableFutures.unwrapCompletionException(throwable);

        if (cause instanceof ThreadLocalTimeoutException) {
            return EStockpileStatusCode.DEADLINE_EXCEEDED;
        }

        if (cause instanceof StockpileRuntimeException) {
            return ((StockpileRuntimeException) cause).getStockpileStatusCode();
        }

        if (cause instanceof UnsupportedFormatRuntimeException) {
            return EStockpileStatusCode.UNSUPPORTED_BINARY_FORMAT;
        }

        if (cause instanceof CorruptedBinaryDataRuntimeException) {
            return EStockpileStatusCode.CORRUPTED_BINARY_DATA;
        }

        if (cause instanceof KikimrKvGenerationChangedRuntimeException) {
            return EStockpileStatusCode.SHARD_ABSENT_ON_HOST;
        }

        return EStockpileStatusCode.INTERNAL_ERROR;
    }
}
