package ru.yandex.stockpile.api.grpc.handler;

import java.time.Instant;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.stockpile.api.DeleteMetricDataRequest;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.api.grpc.TPointAggrGraphDataIterator;
import ru.yandex.stockpile.server.shard.StockpileWriteRequest;

import static ru.yandex.stockpile.api.grpc.handler.Validator.ensureLocalIdValid;
import static ru.yandex.stockpile.api.grpc.handler.Validator.ensureMetricArchiveValid;
import static ru.yandex.stockpile.api.grpc.handler.Validator.ensureValid;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class LogEntryAppender {
    private LogEntryAppender() {
    }

    public static void appendDelete(StockpileWriteRequest.Builder builder, DeleteMetricRequest request) {
        ensureLocalIdValid(request.getMetricId());
        final MetricId metricId = request.getMetricId();
        var archive = builder.archiveRef(metricId.getLocalId());
        archive.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
    }

    public static void appendDeleteData(StockpileWriteRequest.Builder builder, DeleteMetricDataRequest request) {
        ensureLocalIdValid(request.getMetricId());
        if (request.getFromMillis() > 0) {
            throw new StockpileRuntimeException(EStockpileStatusCode.INVALID_REQUEST,
                "Delete data from point in time not supported yet, specified from time " + Instant.ofEpochMilli(request.getFromMillis())
            );
        }

        final MetricId metricId = request.getMetricId();
        if (request.getToMillis() > 0) {
            builder.archiveRef(metricId.getLocalId()).setDeleteBefore(request.getToMillis());
        }
    }

    public static void appendWrite(StockpileWriteRequest.Builder builder, TWriteRequest request) {
        ensureValid(request);
        MetricArchiveMutable archive = builder.archiveRef(request.getMetricId().getLocalId());
        fillHeaders(archive);
        archive.setOwnerShardId(request.getOwnerShardId());
        archive.setType(getType(request));
        archive.addAllFrom(new TPointAggrGraphDataIterator(request.getColumnMask(), request.getPointsList()));
    }

    public static void appendCompressedWrite(StockpileWriteRequest.Builder builder, TCompressedWriteRequest request) {
        ensureLocalIdValid(request.getMetricId());
        StockpileFormat format = StockpileFormat.byNumber(request.getBinaryVersion());
        long localId = request.getMetricId().getLocalId();
        for (TimeSeries.Chunk chunk : request.getChunksList()) {
            MetricArchiveImmutable archive = MetricArchiveNakedSerializer.serializerForFormatSealed(format)
                .deserializeToEof(new StockpileDeserializer(chunk.getContent()));
            ensureMetricArchiveValid(request.getMetricId(), archive);
            builder.addArchive(localId, archive);
        }
    }

    private static MetricType getType(TWriteRequest request) {
        MetricType type = request.getType();
        if (type == MetricType.METRIC_TYPE_UNSPECIFIED) {
            type = StockpileColumns.typeByMask(request.getColumnMask());
        }
        return type;
    }

    public static void fillHeaders(MetricArchiveMutable archive) {
        archive.setOwnerProjectIdEnum(EProjectId.GOLOVAN);
        archive.setDecimPolicyId(EDecimPolicy.POLICY_5_MIN_AFTER_8_DAYS.getNumber());
    }
}
