package ru.yandex.stockpile.api.grpc.handler;

import java.util.concurrent.CompletableFuture;

import io.grpc.MethodDescriptor;

import ru.yandex.concurrency.limits.actors.Limiter;
import ru.yandex.solomon.codec.archive.MetricArchiveUtils;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.aggregation.AggregateConverters;
import ru.yandex.solomon.model.timeseries.aggregation.TimeseriesSummary;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricData;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TCompressedReadManyResponse;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.read.StockpileReadApi;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.stockpile.api.grpc.handler.RequestValidator.ensureValid;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadCompressedManyHandler extends ShardRequestHandler<TReadManyRequest, TCompressedReadManyResponse> {
    private final StockpileReadApi readApi;

    public ReadCompressedManyHandler(StockpileLocalShards shards, StockpileReadApi readApi, Limiter limiter) {
        super(shards, limiter);
        this.readApi = readApi;
    }

    @Override
    protected CompletableFuture<TCompressedReadManyResponse> unaryCall(StockpileShard shard, TReadManyRequest request) {
        ensureValid(request);
        if (!shard.canServeReads()) {
            return completedFuture(response(EStockpileStatusCode.SHARD_NOT_READY));
        }

        StockpileFormat format = StockpileFormat.byNumberOrNull(request.getBinaryVersion());
        if (format == null) {
            return completedFuture(TCompressedReadManyResponse.newBuilder()
                    .setStatus(EStockpileStatusCode.UNSUPPORTED_BINARY_FORMAT)
                    .setStatusMessage("unknown format: " + request.getBinaryVersion())
                    .build());
        }

        ensureDeadlineNotExceeded(request.getDeadline());
        return readApi.readMany(shard, request)
            .collect(metric -> {
                ensureDeadlineNotExceeded(request.getDeadline());

                MetricData.Builder builder = MetricData.newBuilder();
                builder.setType(metric.getType());

                MetricId metricId = metric.getKey();
                if (metricId != null) {
                    builder.setShardId(metricId.getShardId());
                    builder.setLocalId(metricId.getLocalId());
                }

                AggrGraphDataIterable timeSeries = metric.getTimeseries();
                if (timeSeries != null) {
                    var compressedBuilder = builder.getCompressedBuilder();
                    compressedBuilder.setFormatVersion(format.getFormat());
                    var chunk = MetricArchiveUtils.makeChunk(format, metric, request.getFromMillis(), request.getToMillis());
                    if (chunk != null) {
                        compressedBuilder.addChunks(chunk);
                    }
                }

                TimeseriesSummary summary = metric.getSummary();
                if (summary != null) {
                    AggregateConverters.fillAggregate(builder, metric.getSummary());
                }

                return builder.build();
            })
            .thenApply(list -> TCompressedReadManyResponse.newBuilder()
                    .setStatus(EStockpileStatusCode.OK)
                    .addAllMetrics(list)
                    .build());
    }

    @Override
    protected int shardId(TReadManyRequest request) {
        return request.getShardId();
    }

    @Override
    protected TCompressedReadManyResponse response(EStockpileStatusCode status, String details) {
        return TCompressedReadManyResponse.newBuilder()
                .setStatus(status)
                .setStatusMessage(details)
                .build();
    }

    @Override
    public MethodDescriptor<TReadManyRequest, TCompressedReadManyResponse> descriptor() {
        return StockpileServiceGrpc.getReadCompressedManyMethod();
    }

    @Override
    public EStockpileStatusCode getStatusCode(TCompressedReadManyResponse response) {
        return response.getStatus();
    }
}
