package ru.yandex.stockpile.api.grpc.handler;

import java.util.concurrent.CompletableFuture;

import io.grpc.MethodDescriptor;

import ru.yandex.concurrency.limits.actors.Limiter;
import ru.yandex.solomon.codec.archive.MetricArchiveUtils;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TCompressedReadResponse;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.read.Converters;
import ru.yandex.stockpile.api.read.StockpileReadApi;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.stockpile.api.grpc.handler.RequestValidator.ensureValid;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadCompressedOneHandler extends ShardRequestHandler<TReadRequest, TCompressedReadResponse> {
    private final StockpileReadApi readApi;

    public ReadCompressedOneHandler(StockpileLocalShards shards, StockpileReadApi readApi, Limiter limiter) {
        super(shards, limiter);
        this.readApi = readApi;
    }

    protected CompletableFuture<TCompressedReadResponse> unaryCall(StockpileShard shard, TReadRequest request) {
        ensureValid(request);
        if (!shard.canServeReads()) {
            return completedFuture(response(EStockpileStatusCode.SHARD_NOT_READY));
        }

        StockpileFormat format = StockpileFormat.byNumberOrNull(request.getBinaryVersion());
        if (format == null) {
            return completedFuture(TCompressedReadResponse.newBuilder()
                    .setStatus(EStockpileStatusCode.UNSUPPORTED_BINARY_FORMAT)
                    .setStatusMessage("unknown format: " + request.getBinaryVersion())
                    .build());
        }

        ensureDeadlineNotExceeded(request.getDeadline());
        return readApi.readMany(shard, Converters.makeRequest(request))
            .collect(metric -> {
                ensureDeadlineNotExceeded(request.getDeadline());

                var response = TCompressedReadResponse.newBuilder()
                        .setStatus(EStockpileStatusCode.OK)
                        .setType(metric.getType())
                        .setMetricId(request.getMetricId())
                        .setCookie(request.getCookie())
                        .setBinaryVersion(format.getFormat());

                AggrGraphDataIterable timeSeries = metric.getTimeseries();
                if (timeSeries != null) {
                    var chunk = MetricArchiveUtils.makeChunk(format, metric, request.getFromMillis(), request.getToMillis());
                    if (chunk != null) {
                        response.addChunks(chunk);
                    }
                }

                return response.build();
            })
            .thenApply(list -> list.get(0));
    }

    @Override
    protected int shardId(TReadRequest request) {
        return request.getMetricId().getShardId();
    }

    @Override
    protected TCompressedReadResponse response(EStockpileStatusCode status, String details) {
        return TCompressedReadResponse.newBuilder()
                .setStatus(status)
                .setStatusMessage(details)
                .build();
    }

    @Override
    public MethodDescriptor<TReadRequest, TCompressedReadResponse> descriptor() {
        return StockpileServiceGrpc.getReadCompressedOneMethod();
    }

    @Override
    public EStockpileStatusCode getStatusCode(TCompressedReadResponse response) {
        return response.getStatus();
    }
}
