package ru.yandex.stockpile.api.grpc.handler;

import java.util.concurrent.CompletableFuture;

import io.grpc.MethodDescriptor;

import ru.yandex.concurrency.limits.actors.Limiter;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.aggregation.AggregateConverters;
import ru.yandex.solomon.model.timeseries.aggregation.TimeseriesSummary;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricData;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.TUncompressedReadManyResponse;
import ru.yandex.stockpile.api.read.Converters;
import ru.yandex.stockpile.api.read.StockpileReadApi;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadUncompressedManyHandler extends ShardRequestHandler<TReadManyRequest, TUncompressedReadManyResponse> {
    private final StockpileReadApi readApi;

    public ReadUncompressedManyHandler(StockpileLocalShards shards, StockpileReadApi readApi, Limiter limiter) {
        super(shards, limiter);
        this.readApi = readApi;
    }

    @Override
    protected CompletableFuture<TUncompressedReadManyResponse> unaryCall(StockpileShard shard, TReadManyRequest request) {
        if (!shard.canServeReads()) {
            return CompletableFuture.completedFuture(response(EStockpileStatusCode.SHARD_NOT_READY, shard.getLoadState().name()));
        }

        ensureDeadlineNotExceeded(request.getDeadline());
        return readApi.readMany(shard, request)
            .collect(metric -> {
                ensureDeadlineNotExceeded(request.getDeadline());
                MetricData.Builder builder = MetricData.newBuilder();
                builder.setType(metric.getType());

                MetricId metricId = metric.getKey();
                if (metricId != null) {
                    builder.setShardId(metricId.getShardId());
                    builder.setLocalId(metricId.getLocalId());
                }

                AggrGraphDataIterable timeseries = metric.getTimeseries();
                if (timeseries != null) {
                    builder.setUncompressed(Converters.toProto(metric.getTimeseries()));
                }

                TimeseriesSummary summary = metric.getSummary();
                if (summary != null) {
                    AggregateConverters.fillAggregate(builder, metric.getSummary());
                }

                return builder.build();
            })
            .thenApply(metrics -> TUncompressedReadManyResponse.newBuilder()
                    .setStatus(EStockpileStatusCode.OK)
                    .addAllMetrics(metrics)
                    .build());
    }

    @Override
    protected int shardId(TReadManyRequest request) {
        return request.getShardId();
    }

    @Override
    protected TUncompressedReadManyResponse response(EStockpileStatusCode status, String details) {
        return TUncompressedReadManyResponse.newBuilder()
                .setStatus(status)
                .setStatusMessage(details)
                .build();
    }

    @Override
    public MethodDescriptor<TReadManyRequest, TUncompressedReadManyResponse> descriptor() {
        return StockpileServiceGrpc.getReadUncompressedManyMethod();
    }

    @Override
    public EStockpileStatusCode getStatusCode(TUncompressedReadManyResponse response) {
        return response.getStatus();
    }
}
