package ru.yandex.stockpile.api.grpc.handler;

import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import com.google.common.collect.Iterables;
import io.grpc.MethodDescriptor;

import ru.yandex.concurrency.limits.actors.Limiter;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.TReadResponse;
import ru.yandex.stockpile.api.read.Converters;
import ru.yandex.stockpile.api.read.StockpileReadApi;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadUncompressedOneHandler extends ShardRequestHandler<TReadRequest, TReadResponse> {
    private final StockpileReadApi readApi;

    public ReadUncompressedOneHandler(StockpileLocalShards shards, StockpileReadApi readApi, Limiter limiter) {
        super(shards, limiter);
        this.readApi = readApi;
    }

    @Override
    public CompletableFuture<TReadResponse> unaryCall(StockpileShard shard, TReadRequest request) {
        if (!shard.canServeReads()) {
            return CompletableFuture.completedFuture(response(EStockpileStatusCode.SHARD_NOT_READY, shard.getLoadState().name()));
        }

        ensureDeadlineNotExceeded(request.getDeadline());
        return readApi.readMany(shard, Converters.makeRequest(request))
            .collect(Function.identity())
            .thenApply(list -> {
                ensureDeadlineNotExceeded(request.getDeadline());
                Metric<MetricId> metric = Iterables.getOnlyElement(list);

                AggrGraphDataIterable timeseries = metric.getTimeseries() != null
                        ? metric.getTimeseries()
                        : AggrGraphDataArrayList.empty();

                return TReadResponse.newBuilder()
                        .setStatus(EStockpileStatusCode.OK)
                        .setMetricId(request.getMetricId())
                        .setType(metric.getType())
                        .setCookie(request.getCookie())
                        .setColumnMask(timeseries.isEmpty() ? 0 : timeseries.columnSetMask())
                        .addAllPoints(Converters.toProto(timeseries).getPointsList())
                        .build();
            });
    }

    @Override
    protected int shardId(TReadRequest request) {
        return request.getMetricId().getShardId();
    }

    @Override
    protected TReadResponse response(EStockpileStatusCode status, String details) {
        return TReadResponse.newBuilder()
                .setStatus(status)
                .setStatusMessage(details)
                .build();
    }

    @Override
    public MethodDescriptor<TReadRequest, TReadResponse> descriptor() {
        return StockpileServiceGrpc.getReadOneMethod();
    }

    @Override
    public EStockpileStatusCode getStatusCode(TReadResponse response) {
        return response.getStatus();
    }
}
