package ru.yandex.stockpile.api.grpc.handler;

import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

/**
 * @author Vladimir Gordiychuk
 */
public class RequestValidator {

    public static void ensureValid(TReadRequest request) {
        ensureFromTsMillis(request.getFromMillis());
        ensureToMillis(request.getToMillis());
        ensureMetricIdValid(request.getMetricId());
    }

    public static void ensureValid(TReadManyRequest request) {
        ensureFromTsMillis(request.getFromMillis());
        ensureToMillis(request.getToMillis());
        StockpileShardId.validate(request.getShardId());
        for (long localId : request.getLocalIdsList()) {
            StockpileLocalId.validate(localId);
        }
    }

    public static void ensureMetricIdValid(MetricId id) {
        StockpileShardId.validate(id.getShardId());
        StockpileLocalId.validate(id.getLocalId());
    }

    public static void ensureFromTsMillis(long tsMillis) {
        if (tsMillis != 0 && !InstantUtils.isGoodMillis(tsMillis)) {
            throw invalid("From millis %s should be 0 or into range [%s; %s]", tsMillis, InstantUtils.NOT_BEFORE, InstantUtils.NOT_AFTER);
        }
    }

    public static void ensureToMillis(long tsMillis) {
        if (tsMillis != 0 && !InstantUtils.isGoodMillis(tsMillis)) {
            throw invalid("To millis %s should be 0 or into range [%s; %s]", tsMillis, InstantUtils.NOT_BEFORE, InstantUtils.NOT_AFTER);
        }
    }

    private static StockpileRuntimeException invalid(String message, Object... args) {
        throw new StockpileRuntimeException(EStockpileStatusCode.INVALID_REQUEST, String.format(message, args), false);
    }
}
