package ru.yandex.stockpile.api.grpc.handler;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import io.grpc.MethodDescriptor;

import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TServerStatusRequest;
import ru.yandex.stockpile.api.TServerStatusResponse;
import ru.yandex.stockpile.api.TShardStatus;
import ru.yandex.stockpile.server.data.index.stats.IndexStatsData;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

import static ru.yandex.stockpile.api.grpc.handler.Handlers.classifyError;
import static ru.yandex.stockpile.api.grpc.handler.Handlers.logError;

/**
 * @author Vladimir Gordiychuk
 */
public class ServerStatusHandler implements Handler<TServerStatusRequest, TServerStatusResponse> {
    private final StockpileLocalShards shards;

    public ServerStatusHandler(StockpileLocalShards shards) {
        this.shards = shards;
    }

    @Override
    public CompletableFuture<TServerStatusResponse> unaryCall(TServerStatusRequest request) {
        return CompletableFuture.completedFuture(status(request));
    }

    private TServerStatusResponse status(TServerStatusRequest request) {
        try {
            List<TShardStatus> perShardStatus = shards.stream()
                .filter(shard -> shard.canServeReads() || shard.canServeWrites())
                .map(shard -> {
                    IndexStatsData indexStats = shard.indexStats().getTotalByLevel().getTotalByProjects().getTotalByKinds();
                    return TShardStatus.newBuilder()
                        .setShardId(shard.shardId)
                        .setReady(shard.getLoadState() == StockpileShard.LoadState.DONE)
                        .setReadyRead(shard.canServeReads())
                        .setReadyWrite(shard.canServeWrites())
                        .setRecordCount(indexStats.records)
                        .setMetricCount(indexStats.metrics)
                        .build();
                })
                .collect(Collectors.toList());

            return TServerStatusResponse.newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .setOlderSupportBinaryVersion(StockpileFormat.MIN.getFormat())
                .setLatestSupportBinaryVersion(StockpileFormat.CURRENT.getFormat())
                .setTotalShardCount(shards.totalShardsCount())
                .addAllShardStatus(perShardStatus)
                .build();
        } catch (Throwable e) {
            logError(this, request, e);
            EStockpileStatusCode code = classifyError(e);

            return TServerStatusResponse.newBuilder()
                .setStatus(code)
                .setStatusMessage(e.getMessage())
                .build();
        }
    }

    @Override
    public MethodDescriptor<TServerStatusRequest, TServerStatusResponse> descriptor() {
        return StockpileServiceGrpc.getServerStatusMethod();
    }

    @Override
    public EStockpileStatusCode getStatusCode(TServerStatusResponse response) {
        return response.getStatus();
    }
}
