package ru.yandex.stockpile.api.grpc.handler;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Throwables;

import ru.yandex.concurrency.limits.actors.Limiter;
import ru.yandex.concurrency.limits.actors.LimiterNoop;
import ru.yandex.concurrency.limits.actors.OperationStatus;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.stockpile.api.grpc.handler.Handlers.classifyError;
import static ru.yandex.stockpile.api.grpc.handler.Handlers.logError;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class ShardRequestHandler<ReqT, RespT> implements Handler<ReqT, RespT> {
    private final StockpileLocalShards shards;
    private final Limiter limiter;

    public ShardRequestHandler(StockpileLocalShards shards) {
        this(shards, LimiterNoop.INSTANCE);
    }

    public ShardRequestHandler(StockpileLocalShards shards, Limiter limiter) {
        this.shards = shards;
        this.limiter = limiter;
    }

    protected abstract CompletableFuture<RespT> unaryCall(StockpileShard shard, ReqT request);

    protected abstract int shardId(ReqT request);

    protected abstract RespT response(EStockpileStatusCode status, String details);

    protected RespT response(EStockpileStatusCode status) {
        return response(status, "");
    }

    @Override
    public CompletableFuture<RespT> unaryCall(ReqT request) {
        int shardId = shardId(request);
        if (shardId == 0) {
            var response = response(EStockpileStatusCode.INVALID_REQUEST, "Unknown shardId");
            return CompletableFuture.completedFuture(response);
        }

        var shard = shards.getShardById(shardId);
        if (shard == null) {
            return CompletableFuture.completedFuture(response(EStockpileStatusCode.SHARD_ABSENT_ON_HOST, ""));
        }

        var permit = limiter.acquire();
        if (permit == null) {
            limiter.addStatus(OperationStatus.REJECT);
            return completedFuture(response(EStockpileStatusCode.RESOURCE_EXHAUSTED, "too many inflight requests"));
        }

        long startNanos = System.nanoTime();
        try {
            return safeCall(shard, request)
                    .whenComplete((resp, e) -> {
                        if (e != null) {
                            permit.release(toOperationStatus(classifyError(e)));
                        } else {
                            permit.release(toOperationStatus(getStatusCode(resp)));
                        }
                    });
        } finally {
            shard.metrics.utimeNanos.mark(System.nanoTime() - startNanos);
        }
    }

    private OperationStatus toOperationStatus(EStockpileStatusCode code) {
        switch (code) {
            case OK:
                return OperationStatus.SUCCESS;
            case DEADLINE_EXCEEDED:
                return OperationStatus.DROP;
            default:
                return OperationStatus.IGNORE;
        }
    }

    private CompletableFuture<RespT> safeCall(StockpileShard shard, ReqT request) {
        final int shardId = shard.shardId;
        try {
            return unaryCall(shard, request)
                    .exceptionally(e -> onError(shardId, e));
        } catch (Throwable e) {
            return completedFuture(onError(shardId, e));
        }
    }

    private RespT onError(int shardId, Throwable e) {
        logError(this, StockpileShardId.toString(shardId), e);
        EStockpileStatusCode code = classifyError(e);

        return response(code, Throwables.getStackTraceAsString(e));
    }

    protected boolean isDeadlineExceeded(long deadline) {
        return deadline != 0 && System.currentTimeMillis() >= deadline;
    }

    protected void ensureDeadlineNotExceeded(long deadline) {
        if (isDeadlineExceeded(deadline)) {
            throw new StockpileRuntimeException(EStockpileStatusCode.DEADLINE_EXCEEDED, Instant.ofEpochMilli(deadline).toString());
        }
    }
}
