package ru.yandex.stockpile.api.grpc.handler;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

/**
 * @author Vladimir Gordiychuk
 */
class Validator {
    private Validator() {
    }

    static void ensureValid(TWriteRequest request) {
        ensureLocalIdValid(request.getMetricId());

        final MetricType type;
        if (request.getType() != MetricType.METRIC_TYPE_UNSPECIFIED) {
            type = request.getType();
        } else {
            type = StockpileColumns.typeByMask(request.getColumnMask());
        }

        if (request.getColumnMask() == 0) {
            throw invalid("Column mask not specified: " + request.getColumnMask());
        }

        if (!isSupportKind(type)) {
            throw invalid("Not supported time series type: %s", type);
        }

        for (TPoint point : request.getPointsList()) {
            ensurePointValid(request.getMetricId(), type, point);
        }
    }

    static void ensurePointValid(MetricId metricId, MetricType type, TPoint point) {
        ensureTsValid(metricId, point.getTimestampsMillis());
        if (type == MetricType.HIST || type == MetricType.HIST_RATE) {
            var histogram = point.getHistogram();
            if (histogram.getBoundsCount() != histogram.getBucketsCount()) {
                throw invalid("%s: boundsSize(%s) != bucketsSize(%s) at %s",
                        metricId,
                        histogram.getBoundsCount(),
                        histogram.getBucketsCount(),
                        point.getTimestampsMillis());
            }

            if (histogram.getBucketsCount() > Histograms.MAX_BUCKETS_COUNT) {
                throw invalid("%s: bucketsCount must <= %s, got: %s",
                        metricId,
                        Histograms.MAX_BUCKETS_COUNT,
                        histogram.getBucketsCount());
            }
        }
    }

    static void ensureTsValid(MetricId metricId, long tsMillis) {
        if (!InstantUtils.isGoodMillis(tsMillis)) {
            throw invalid("%s: invalid timestamp %s", metricId, tsMillis);
        }
    }

    static void ensureMetricArchiveValid(MetricId metricId, MetricArchiveImmutable archive) {
        AggrGraphDataListIterator iterator = archive.iterator();
        var point = RecyclableAggrPoint.newInstance();
        try {
            while (iterator.next(point)) {
                ensureTsValid(metricId, point.tsMillis);
            }
        } finally {
            point.recycle();
        }
    }

    static void ensureLocalIdValid(MetricId metricId) {
        // Use legacy valid method because still exists metrics with legacy local id that not pass new check
        if (!StockpileLocalId.isValid(metricId.getLocalId())) {
            throw invalid("Not valid local id for metric id: " + metricId);
        }
    }

    private static boolean isSupportKind(MetricType type) {
        switch (type) {
            case METRIC_TYPE_UNSPECIFIED:
            case COUNTER:
            case DGAUGE:
            case ISUMMARY:
            case DSUMMARY:
            case HIST:
            case RATE:
            case IGAUGE:
            case HIST_RATE:
            case LOG_HISTOGRAM:
                return true;
            default:
                return false;
        }
    }

    private static StockpileRuntimeException invalid(String message, Object... args) {
        throw new StockpileRuntimeException(EStockpileStatusCode.INVALID_REQUEST, String.format(message, args));
    }
}
