package ru.yandex.stockpile.api.grpc.handler;

import java.util.concurrent.CompletableFuture;

import io.grpc.MethodDescriptor;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TCompressedWriteResponse;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;
import ru.yandex.stockpile.server.shard.StockpileWriteRequest;

import static ru.yandex.stockpile.api.grpc.handler.Validator.ensureMetricArchiveValid;

/**
 * @author Vladimir Gordiychuk
 */
public class WriteCompressedOneHandler extends ShardRequestHandler<TCompressedWriteRequest, TCompressedWriteResponse> {

    public WriteCompressedOneHandler(StockpileLocalShards shards) {
        super(shards);
    }

    @Override
    public CompletableFuture<TCompressedWriteResponse> unaryCall(StockpileShard shard, TCompressedWriteRequest request) {
        return shard.pushBatch(parse(request))
            .thenApply(result -> response(EStockpileStatusCode.OK));
    }

    @Override
    protected int shardId(TCompressedWriteRequest request) {
        return request.getMetricId().getShardId();
    }

    @Override
    protected TCompressedWriteResponse response(EStockpileStatusCode status, String details) {
        return TCompressedWriteResponse.newBuilder()
                .setStatus(status)
                .setStatusMessage(details)
                .build();
    }

    @Override
    public MethodDescriptor<TCompressedWriteRequest, TCompressedWriteResponse> descriptor() {
        return StockpileServiceGrpc.getWriteCompressedOneMethod();
    }

    @Override
    public EStockpileStatusCode getStatusCode(TCompressedWriteResponse response) {
        return response.getStatus();
    }

    private StockpileWriteRequest parse(TCompressedWriteRequest request) {
        final StockpileFormat format = StockpileFormat.byNumber(request.getBinaryVersion());
        final MetricId metricId = request.getMetricId();

        StockpileWriteRequest.Builder builder = StockpileWriteRequest.newBuilder();
        for (TimeSeries.Chunk chunk : request.getChunksList()) {
            MetricArchiveImmutable archive = MetricArchiveNakedSerializer.serializerForFormatSealed(format)
                    .deserializeToEof(new StockpileDeserializer(chunk.getContent()));
            ensureMetricArchiveValid(metricId, archive);
            builder.addArchive(metricId.getLocalId(), archive);
        }
        return builder.build();
    }
}
