package ru.yandex.stockpile.api.grpc.handler;

import java.util.concurrent.CompletableFuture;

import io.grpc.MethodDescriptor;
import io.grpc.protobuf.ProtoUtils;

import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TWriteDataBinaryResponse;
import ru.yandex.stockpile.api.grpc.handler.parse.WriteDataBinaryMarshaller;
import ru.yandex.stockpile.api.grpc.handler.parse.WriteDataBinaryRequest;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;
import ru.yandex.stockpile.server.shard.StockpileWriteRequest;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class WriteDataBinaryHandler extends ShardRequestHandler<WriteDataBinaryRequest, TWriteDataBinaryResponse> {
    private static final io.grpc.MethodDescriptor<WriteDataBinaryRequest, TWriteDataBinaryResponse> ENDPOINT =
        MethodDescriptor.<WriteDataBinaryRequest, TWriteDataBinaryResponse>newBuilder()
            .setType(MethodDescriptor.MethodType.UNARY)
            .setFullMethodName(StockpileServiceGrpc.getWriteDataBinaryMethod().getFullMethodName())
            .setRequestMarshaller(new WriteDataBinaryMarshaller())
            .setResponseMarshaller(ProtoUtils.marshaller(TWriteDataBinaryResponse.getDefaultInstance()))
            .build();

    public WriteDataBinaryHandler(StockpileLocalShards shards) {
        super(shards);
    }

    @Override
    public CompletableFuture<TWriteDataBinaryResponse> unaryCall(StockpileShard shard, WriteDataBinaryRequest request) {
        shard.metrics.utimeNanos.mark(request.parseTimeNanos);
        if (!shard.canServeWrites()) {
            request.release();
            return completedFuture(response(EStockpileStatusCode.SHARD_NOT_READY, shard.getLoadState().name()));
        }

        return shard.pushBatch(new StockpileWriteRequest(request.content, null, 0, 0))
            .thenApply(txn -> response(EStockpileStatusCode.OK));
    }

    @Override
    protected int shardId(WriteDataBinaryRequest request) {
        return request.shardId;
    }

    @Override
    protected TWriteDataBinaryResponse response(EStockpileStatusCode status, String details) {
        return TWriteDataBinaryResponse.newBuilder()
                .setStatus(status)
                .setStatusMessage(details)
                .build();
    }

    @Override
    public MethodDescriptor<WriteDataBinaryRequest, TWriteDataBinaryResponse> descriptor() {
        return ENDPOINT;
    }

    @Override
    public EStockpileStatusCode getStatusCode(TWriteDataBinaryResponse response) {
        return response.getStatus();
    }
}
